import click
import asyncio

from crm.agency_cabinet.grants.server.src import run_server as _run_server
from crm.agency_cabinet.grants.server.src.celery import celery
from crm.agency_cabinet.grants.server.bin.scripts.generate_roles_and_permissions import (
    add_role_for_agency,
    add_permission as _add_permission,
)


@click.group()
def cli():
    pass


@cli.command('run_server')
def run_server():
    return _run_server()


@cli.command('celery_beat', context_settings=dict(
    ignore_unknown_options=True,
))
@click.argument('celery_beat_args', nargs=-1, type=click.UNPROCESSED)
def celery_beat(celery_beat_args):
    return celery.start(['beat'] + list(celery_beat_args))


@cli.command('celery_worker', context_settings=dict(
    ignore_unknown_options=True,
))
@click.argument('celery_worker_args', nargs=-1, type=click.UNPROCESSED)
def celery_worker(celery_worker_args):
    return celery.worker_main(argv=['worker'] + list(celery_worker_args))


@cli.command('celery', context_settings=dict(
    ignore_unknown_options=True,
))
@click.argument('celery_cli_args', nargs=-1, type=click.UNPROCESSED)
def celery_cli(celery_cli_args):
    if celery_cli_args and celery_cli_args[0] == 'tasks':
        for task_name in celery.tasks.keys():
            print(task_name)
        return
    return celery.start(argv=['call'] + list(celery_cli_args))


@cli.command('shell')
def shell():
    from IPython import embed
    embed()


@cli.command('add_role')
@click.argument('agency_id', type=int)
@click.argument('role_name', type=str)
@click.option('--editable_permission', '-p', multiple=True, type=str, default=[])
@click.option('--non_editable_permission', '-np', multiple=True, type=str, default=[])
@click.option('--description', '-d', type=str, default=None)
def add_role(agency_id, role_name, description, editable_permission, non_editable_permission):
    try:
        return asyncio.run(add_role_for_agency(
            agency_id,
            role_name,
            description,
            [(p, True) for p in editable_permission] + [(p, False) for p in non_editable_permission]
        ))
    except ValueError as ex:
        if str(ex) == 'ALREADY_EXISTS':
            raise click.UsageError(f'Role {role_name} for agency {agency_id} already exists')
        elif str(ex) == 'UNKNOWN_AGENCY_ID':
            raise click.UsageError(f'Unknown agency_id {agency_id}')


@cli.command('add_permission')
@click.argument('permission_name', type=str)
@click.option('--description', '-d', type=str, default=None)
def add_permission(permission_name, description):
    return asyncio.run(_add_permission(
        permission_name,
        description
    ))


def main():
    cli()


if __name__ == '__main__':
    main()
