import typing
import logging
from sqlalchemy import and_

from crm.agency_cabinet.agencies.common import structs
from crm.agency_cabinet.grants.common.consts import Permissions, Roles, ROLE_TO_DESCRIPTION
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.grants.server.src.db.models import Permission, RolesPermissionsMap, UserRole, PartnersRolesMap, \
    UsersRolesPermissionsMap, Partner
from crm.agency_cabinet.grants.server.src.db import db
from crm.agency_cabinet.grants.server.src.config import GrantsConfig
from crm.agency_cabinet.grants.server.bin.scripts.common import get_bind_context


LOGGER = logging.getLogger('scripts.generate_roles_and_permissions')


PERMISSION_DESCRIPTION_MAP = {
    Permissions.certification.value: 'Доступ к странице Сертификация',
    Permissions.rewards.value: 'Доступ к странице Премии',
    Permissions.client_bonuses.value: 'Доступ к странице Клиентские бонусы',
    Permissions.documents.value: 'Доступ к странице Документы',
    Permissions.roles.value: 'Доступ к странице Роли'
}


async def add_permission(key, description):
    async with get_bind_context():
        return await Permission.get_or_create(
            {'name': key},
            {'name': key, 'description': description}
        )


async def generate_default_permissions():
    async with get_bind_context():
        for permission in Permissions:
            await add_permission(permission.value, PERMISSION_DESCRIPTION_MAP.get(permission.value))


async def add_role_for_agency(agency_id: int,
                              role_name: str, description: str = None,
                              permissions_list: typing.List[typing.Tuple[str, bool]] = (),
                              check_agency_id=True):
    async with get_bind_context():
        async with db.transaction(read_only=False, reuse=False):
            if check_agency_id:
                cfg = GrantsConfig.from_environ()
                sd = ServiceDiscovery(cfg.amqp_url)
                async with sd:
                    res = await sd.agencies.get_agencies_info([agency_id])
                    if not res:
                        raise ValueError('UNKNOWN_AGENCY_ID')
            partner = Partner.query.where(Partner.external_id == str(agency_id)).first()
            if partner is None:
                raise ValueError('ACCOUNT_NOT_FOUND')
            duplicate = await db.select(
                [
                    UserRole.id
                ],
            ).select_from(UserRole.join(PartnersRolesMap)).where(
                and_(
                    PartnersRolesMap.agency_id == agency_id,
                    UserRole.name == role_name
                )
            ).gino.first()
            if duplicate is not None:
                raise ValueError('ALREADY_EXISTS')
            role_model = await UserRole.create(
                name=role_name,
                description=description
            )
            await PartnersRolesMap.create(
                agency_id=agency_id,
                role_id=role_model.id,
                partner_id=partner.id
            )
            all_permissions = {permission.name: permission for permission in await Permission.query.gino.all()}
            for permission, is_editable in permissions_list:
                permission_model = all_permissions.get(permission)
                if permission_model is None:
                    LOGGER.warning('Can\'t find permissions with name %s', permission)
                await RolesPermissionsMap.create(
                    permission_id=permission_model.id,
                    role_id=role_model.id,
                    is_editable=is_editable
                )


async def generate_default_roles_for_agency(agency_id: int, agency_info_map: typing.Dict[int, structs.AgencyInfo]):
    async with get_bind_context():
        async with db.transaction(read_only=False, reuse=False):
            all_default_permissions = await Permission.query.where(
                Permission.name.in_((permission.value for permission in Permissions))
            ).gino.all()
            manager_roles_list = [(permission.name, True) for permission in all_default_permissions if permission.name != Permissions.roles.value]
            admin_roles_list = [(permission.name, False) for permission in all_default_permissions]
            for role in Roles:
                permission_list = []
                if role == Roles.manager:
                    permission_list = manager_roles_list
                elif role == Roles.owner or role == Roles.admin:
                    permission_list = admin_roles_list
                try:
                    await add_role_for_agency(
                        agency_id,
                        role.value,
                        ROLE_TO_DESCRIPTION[role.value].format(agency_info_map[agency_id].name),
                        permission_list,
                        check_agency_id=False
                    )
                except Exception as ex:
                    LOGGER.warning('Error during %s role creation for agency %s: %s', role.value, agency_id, ex)


async def generate_default_roles_for_all_agencies():
    cfg = GrantsConfig.from_environ()
    sd = ServiceDiscovery(cfg.amqp_url)
    async with sd, get_bind_context():
        agencies_info_map = {
            agency_info.agency_id: agency_info for agency_info in await sd.agencies.get_all_agencies_info()
        }
        for agency_id in agencies_info_map.keys():
            try:
                await generate_default_roles_for_agency(agency_id, agencies_info_map)
            except Exception as ex:
                LOGGER.warning('Error during default roles creation for agency %s: %s', agency_id, ex)


async def add_new_permission_to_role(
    permission_name: str,
    is_editable: bool,
    is_active: bool = True,
    agency_id: typing.Optional[int] = None,
    role_name: typing.Optional[str] = None
):
    async with get_bind_context():
        async with db.transaction(read_only=False, reuse=False):
            permission = Permissions(permission_name)
            permission = await add_permission(permission.value, PERMISSION_DESCRIPTION_MAP.get(permission.value))
            q = db.select(
                [
                    UserRole.id,
                    UserRole.name.label('role_name'),
                    PartnersRolesMap.agency_id
                ],
            ).select_from(UserRole.join(PartnersRolesMap))

            if agency_id is not None:
                q = q.where(PartnersRolesMap.agency_id == agency_id)

            if role_name is not None:
                q = q.where(UserRole.name == role_name)
            roles_to_change = await q.gino.all()

            for row in roles_to_change:
                duplicate = await db.select(
                    [
                        RolesPermissionsMap.role_id
                    ]
                ).select_from(RolesPermissionsMap).where(and_(RolesPermissionsMap.permission_id == permission.id, RolesPermissionsMap.role_id == row.id)).gino.first()
                if duplicate is not None:
                    # TODO: change is_editable if it differs
                    continue

                role_permission_map = await RolesPermissionsMap.create(
                    permission_id=permission.id,
                    role_id=row.id,
                    is_editable=is_editable
                )
                users_to_change = await db.select(
                    [
                        db.func.distinct(UsersRolesPermissionsMap.user_id).label('user_id'),

                    ]
                ).select_from(UsersRolesPermissionsMap).where(UsersRolesPermissionsMap.role_id == row.id).gino.all()
                for user_row in users_to_change:
                    await UsersRolesPermissionsMap.create(
                        permission_id=role_permission_map.permission_id,
                        role_id=role_permission_map.role_id,
                        user_id=user_row.user_id,
                        is_editable=is_editable,
                        is_active=is_active
                    )
