import asyncio
import logging.config
from crm.agency_cabinet.common.server.common.logging_config import get_logging_config
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.grants.common import QUEUE
from crm.agency_cabinet.grants.server.src.handler import Handler
from crm.agency_cabinet.grants.server.src.config import GrantsConfig, Settings
from crm.agency_cabinet.grants.server.src.db.engine import db
from crm.agency_cabinet.common.blackbox import BlackboxClient
from gino.dialects.asyncpg import Pyscopg2Pool
from crm.agency_cabinet.common.monitoring.utils.server import run_ping_service
from crm.agency_cabinet.common.monitoring.utils.registry import setup_monitoring


__all__ = ['run_server']


def run_server():
    cfg = GrantsConfig.from_environ()
    logging.config.dictConfig(get_logging_config(cfg.verbose))

    asyncio.run(_run_server(cfg))


async def _run_server(cfg: GrantsConfig):
    await db.set_bind(bind=str(cfg.db.get_dsn()), pool_class=Pyscopg2Pool, ssl=Settings.db['ssl'])

    blackbox_client = BlackboxClient(cfg.tvm2, cfg.blackbox)
    metric_registry = setup_monitoring()

    async with ServiceDiscovery(cfg.amqp_url, with_rpc_client=False) as sd:  # type: ServiceDiscovery
        handler = Handler(blackbox_client, metric_registry=metric_registry)

        async with sd.rpc_server(QUEUE) as server:
            await asyncio.gather(server.drain(handler), run_ping_service(metric_registry))
