import logging

from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder
from yt.wrapper import YtClient

from crm.agency_cabinet.common.yt.loaders import YqlLoader
from crm.agency_cabinet.rewards.common.consts import CONTRACT_CODES_FOR_ALL

from .synchronizer import AgencyPartnersSynchronizer


LOGGER = logging.getLogger('celery.load_users')


class AgencyPartnersLoader(YqlLoader):
    direct_dump_root = '//home/direct/mysql-sync/current'
    contracts_table_path = '//home/balance/prod/bo/v_contract_apex_full'

    YQL_QUERY = '''
        use hahn;

        DECLARE $direct_dump_root AS String;
        DECLARE $contracts_table_path AS String;
        DECLARE $contracts_codes AS List<String>;

        $contracts = (
            SELECT
                v_contract_apex_full.agency_id AS agency_id,
                AGGREGATE_LIST(
                        AsStruct(
                        `agency_name` as agency_name,
                        `finish_dt` as finish_dt
                    )
                ) as contracts_list
            FROM $contracts_table_path as v_contract_apex_full
            WHERE wsale_ag_prm_awrd_sc_tp IN $contracts_codes
            and v_contract_apex_full.finish_dt >= '2022-03-01'
            group by agency_id
        );

        SELECT
            clients.ClientID as agency_id,
            clients.name as agency_name,
            contracts.contracts_list
        FROM LIKE(
            $direct_dump_root,
            `ppc:%`,
            `straight/clients`
        ) AS clients
        LEFT JOIN $contracts as contracts on contracts.agency_id = clients.ClientID
        where clients.role = 'agency';
    '''

    def __init__(self, yt_client: YtClient, yql_client: YqlClient):
        super().__init__(
            yt_client, yql_client,
            yql_parameters={
                '$direct_dump_root': YqlParameterValueBuilder.make_string(self.direct_dump_root),
                '$contracts_table_path': YqlParameterValueBuilder.make_string(self.contracts_table_path),
                '$contracts_codes': YqlParameterValueBuilder.make_list([
                    YqlParameterValueBuilder.make_string(code) for code in CONTRACT_CODES_FOR_ALL
                ]),
            },
            table_paths=[self.contracts_table_path]
        )

    async def _is_should_execute(self):
        return True

    async def _load(self, *args, **kwargs) -> bool:
        results = await self._get_results()
        agencies = results.table.get_iterator()

        return await AgencyPartnersSynchronizer().process_data(agencies)
