import logging
import typing

from sqlalchemy import and_

from crm.agency_cabinet.common.yt.synchronizers import BaseSynchronizer
from crm.agency_cabinet.grants.common.consts import Roles, Permissions
from crm.agency_cabinet.grants.common.consts import PartnerType
from crm.agency_cabinet.grants.server.src.db import db, models
from crm.agency_cabinet.grants.server.src.db.helpers import create_role, RoleSpec


LOGGER = logging.getLogger('celery.load_partners')


AGENCY_ROLES = [
    RoleSpec(
        role=Roles.owner,
        permissions=[Permissions.roles, Permissions.rewards, Permissions.client_bonuses],
        is_editable=False
    ),
    RoleSpec(
        role=Roles.admin,
        permissions=[Permissions.roles, Permissions.rewards, Permissions.client_bonuses],
        is_editable=False
    ),
    RoleSpec(
        role=Roles.manager,
        permissions=[Permissions.rewards, Permissions.client_bonuses],
        is_editable=True
    ),
    RoleSpec(
        role=Roles.suggested_user,
        permissions=[],
        is_editable=False
    ),
]


AGENCY_WO_CONTRACT_ROLES = [
    RoleSpec(
        role=Roles.owner,
        permissions=[Permissions.roles, Permissions.ord],
        is_editable=False
    ),
    RoleSpec(
        role=Roles.admin,
        permissions=[Permissions.roles, Permissions.ord],
        is_editable=False
    ),
    RoleSpec(
        role=Roles.ord,
        permissions=[Permissions.ord],
        is_editable=False
    ),
    RoleSpec(
        role=Roles.suggested_user,
        permissions=[],
        is_editable=False
    ),
]


class BasePartnersSynchronizer(BaseSynchronizer):
    permissions_map = None

    async def _prepare(self):
        permissions = await models.Permission.query.gino.all()
        self.permissions_map = {Permissions(p.name): p.id for p in permissions}

    async def _validate_permissions(self, role_specs: typing.List[RoleSpec]):
        roles_permissions = set()
        for role in role_specs:
            roles_permissions |= set(role.permissions)

        diff = roles_permissions - set(self.permissions_map.keys())
        if diff:
            LOGGER.error('There is not enough permissions: %s', diff)
            raise Exception(f'Unable to run {self.__class__.__name__}')

    async def _create_roles(self, partner_id: int, partner_name: str, role_specs: typing.List[RoleSpec]):
        for spec in role_specs:
            await create_role(partner_id, partner_name, spec, self.permissions_map)


class AgencyPartnersSynchronizer(BasePartnersSynchronizer):

    external_id_shared_types = [
        PartnerType.agency.value,
        PartnerType.agency_wo_contract.value,
        PartnerType.direct_client.value
    ]

    roles_map = {
        PartnerType.agency: AGENCY_ROLES,
        PartnerType.agency_wo_contract: AGENCY_WO_CONTRACT_ROLES
    }

    async def process_data(self, rows: list[tuple], *_, **__) -> bool:
        await self._prepare()
        await self._validate_permissions(AGENCY_ROLES+AGENCY_WO_CONTRACT_ROLES)

        for row in rows:
            async with db.transaction(read_only=False, reuse=False):
                await self._process_partner_data(row[0], row[1], row[2])
        return True

    async def _process_partner_data(self, agency_id, agency_name, contracts):
        agency_id = str(agency_id)
        partner_type = PartnerType.agency if contracts is not None else PartnerType.agency_wo_contract
        if contracts:
            latest_contract = sorted(contracts, key=lambda c: c.finish_dt, reverse=True)[0]
            agency_name = latest_contract.agency_name

        if not agency_name:
            # real agencies always have name
            return

        partner = await models.Partner.query.where(
            and_(
                models.Partner.external_id == agency_id,
                models.Partner.type.in_(self.external_id_shared_types)
            )
        ).gino.first()

        if partner is not None:
            if not partner.name:
                await partner.update(name=agency_name).apply()
            # todo: add roles generation when partner moving from agency_wo_contract/direct_client to agency
            return

        partner = await models.Partner.create(
            name=agency_name,
            external_id=agency_id,
            type=partner_type.value
        )

        await self._create_roles(partner.id, partner.name, self.roles_map[partner_type])
