import logging

from yt.wrapper import YtClient
from yql.api.v1.client import YqlClient

from crm.agency_cabinet.common.celery.base import async_to_sync
from crm.agency_cabinet.common.celery.utils import extract_shard_names
from crm.agency_cabinet.grants.server.src import config
from crm.agency_cabinet.grants.server.src.celery.base import celery_app as celery


from .loader import UsersLoader


LOGGER = logging.getLogger('celery.load_users_task')

BASE_SOURCE_PATH = '//home/direct/mysql-sync/current'


@celery.task(bind=True)
def load_users_single_shard_task(self, clients_table_path, users_table_path, force_load=False):
    @async_to_sync
    async def _load_users():
        client_config = {
            'cluster': 'hahn',
            'token': config.YT_CONFIG['TOKEN'],
            'config': {}
        }

        yt_client = YtClient(
            proxy=client_config['cluster'],
            token=client_config['token'],
            config=client_config.get('config')
        )

        yql_client = YqlClient(token=config.YQL_CONFIG['TOKEN'])

        users_loader = UsersLoader(yt_client, yql_client, clients_table_path, users_table_path)
        return await users_loader.load(force_load=force_load)

    result = _load_users()
    LOGGER.info('Finished sync for UsersLoder with params: %s and %s. Success: %s', clients_table_path, users_table_path, result)


@celery.task(bind=True)
def load_users_task(self, force_load=False):
    client_config = {
        'cluster': 'hahn',
        'token': config.YT_CONFIG['TOKEN'],
        'config': {}
    }

    for shard in extract_shard_names(base_path=BASE_SOURCE_PATH, client_config=client_config):
        clients_path = f'{shard}/straight/clients'
        users_path = f'{shard}/straight/users'

        load_users_single_shard_task.delay(
            clients_table_path=clients_path,
            users_table_path=users_path,
            force_load=force_load
        )
