import logging
import typing

from sqlalchemy import and_

from crm.agency_cabinet.common.yt.synchronizers import BaseSynchronizer
from crm.agency_cabinet.grants.common.consts import Roles, PartnerType
from crm.agency_cabinet.grants.server.src.db import db, models
from crm.agency_cabinet.grants.server.src.db.queries import get_partner_roles


LOGGER = logging.getLogger('celery.load_users')


CHIEF_REP_TYPE = 'chief'


class UsersSynchronizer(BaseSynchronizer):

    async def process_data(self, rows: list[tuple],  *args, **kwargs) -> bool:
        for row in rows:
            async with db.transaction(read_only=False, reuse=False):
                await self._process_agency_data(row[0], row[1])
        return True

    async def _process_agency_data(self, agency_id, users) -> None:
        partner = await models.Partner.query.where(
            and_(
                models.Partner.external_id == str(agency_id),
                models.Partner.type.in_((PartnerType.agency.value, PartnerType.agency_wo_contract.value))
            )
        ).gino.first()
        if partner is None:
            return

        roles = await get_partner_roles(partner.id)
        if not roles:
            return

        owner_role = next((role for role in roles if role.name == Roles.owner.value), None)
        if owner_role is None:
            LOGGER.error('There is no owner role for agency %s', agency_id)
            return

        suggested_user_role = next((role for role in roles if role.name == Roles.suggested_user.value), None)
        if suggested_user_role is None:
            LOGGER.error('There is no suggested_user role for agency %s', agency_id)
            return

        roles_without_owner = [role for role in roles if role.name != Roles.owner.value]

        chief_user = next((user for user in users if user.rep_type == CHIEF_REP_TYPE), None)
        suggested_users = [user for user in users if user.rep_type != CHIEF_REP_TYPE]

        is_active = partner.type == PartnerType.agency.value
        await self._process_owner(chief_user, owner_role, is_active)
        await self._process_suggested_users(suggested_users, suggested_user_role, roles_without_owner, is_active)

    async def _process_owner(self, chief_user, owner_role, is_active):
        chief_user = await self._get_or_create_user(chief_user)
        user_ids = await self._get_user_ids([owner_role])

        users_to_remove = [user_id for user_id in user_ids if user_id != chief_user.id]
        if users_to_remove:
            await models.UsersRolesPermissionsMap.delete.where(
                and_(
                    models.UsersRolesPermissionsMap.role_id == owner_role.id,
                    models.UsersRolesPermissionsMap.user_id.in_(users_to_remove)
                )
            ).gino.status(read_only=False, reuse=False)

        if chief_user.id in user_ids:
            return

        permission_map = await self._get_role_permission_map(owner_role)
        for row in permission_map:
            await models.UsersRolesPermissionsMap.create(
                user_id=chief_user.id,
                role_id=owner_role.id,
                permission_id=row.permission_id,
                is_editable=row.is_editable,
                is_active=is_active
            )

    async def _process_suggested_users(self, suggested_users, suggested_role, roles_without_owner, is_active):
        users = [await self._get_or_create_user(user_struct) for user_struct in suggested_users]
        current_user_ids = set(await self._get_user_ids(roles_without_owner))

        users_to_add = [user.id for user in users if user.id not in current_user_ids]
        users_to_remove = current_user_ids - set([user.id for user in users])

        if users_to_remove:
            await models.UsersRolesPermissionsMap.delete.where(
                and_(
                    models.UsersRolesPermissionsMap.role_id.in_([role.id for role in roles_without_owner]),
                    models.UsersRolesPermissionsMap.user_id.in_(users_to_remove)
                )
            ).gino.status(read_only=False, reuse=False)

        for user_id in users_to_add:
            await models.UsersRolesPermissionsMap.create(
                user_id=user_id,
                role_id=suggested_role.id,
                permission_id=None,
                is_editable=False,
                is_active=is_active
            )

    @staticmethod
    async def _get_or_create_user(user_struct) -> models.User:
        return await models.User.get_or_create(
            {'yandex_uid': user_struct.uid},
            {
                'yandex_uid': user_struct.uid,
                'email': user_struct.email,
                'login': user_struct.login,
                'display_name': user_struct.fio,
                'type': 'external',
            }
        )

    @staticmethod
    async def _get_user_ids(roles: list[models.Role]) -> list[int]:
        query = db.select([
            db.func.distinct(models.UsersRolesPermissionsMap.user_id).label('user_id')
        ]).select_from(models.UsersRolesPermissionsMap).where(
            models.UsersRolesPermissionsMap.role_id.in_([role.id for role in roles])
        )
        rows = await query.gino.all()
        return [row.user_id for row in rows]

    @staticmethod
    async def _get_role_permission_map(role: models.Role) -> list[typing.Any]:
        query = db.select([
            models.RolesPermissionsMap.permission_id,
            models.RolesPermissionsMap.is_editable
        ]).select_from(models.RolesPermissionsMap).where(
            models.RolesPermissionsMap.role_id == role.id
        )
        return await query.gino.all()
