import typing

from sqlalchemy import and_
from crm.agency_cabinet.grants.server.src.db import models, db


def build_select_permissions(
    partner_id: int,
    user_id: int,
    prefix: str = '',
    is_active: bool = True,
    permissions_ids: list[int] = (),
    with_roles: bool = False
):
    columns = [models.Permission.name.label('name')]
    joins_list = [
        (models.Permission, models.UsersRolesPermissionsMap.permission_id == models.Permission.id),
        (models.PartnersRolesMap, models.PartnersRolesMap.role_id == models.UsersRolesPermissionsMap.role_id),
    ]
    if with_roles:
        columns.extend([
            models.UsersRolesPermissionsMap.role_id,
            models.UserRole.name.label('role_name'),
            models.UserRole.display_name.label('role_display_name')
        ])
        joins_list.append(
            (models.UserRole, models.UsersRolesPermissionsMap.role_id == models.UserRole.id)
        )

    selectable = models.UsersRolesPermissionsMap
    for join_condition in joins_list:
        selectable = selectable.join(*join_condition)

    q = db.select(
        columns
    ).select_from(selectable).where(
        and_(
            models.PartnersRolesMap.partner_id == partner_id,
            models.UsersRolesPermissionsMap.user_id == user_id,
            models.UsersRolesPermissionsMap.is_active.is_(is_active),
        )
    )

    if permissions_ids:
        q = q.where(
            models.UsersRolesPermissionsMap.permission_id.in_(permissions_ids)
        )

    if prefix:
        q = q.where(
            models.Permission.name.ilike(f'{prefix}%')
        )

    return q


def build_select_available_roles(partner_id: int, skip_roles: typing.Iterable[str] = (), role_prefix: str = ''):
    q = db.select(
        [
            models.RolesPermissionsMap.role_id,
            models.UserRole.name.label('role_name'),
            models.UserRole.display_name.label('role_display_name'),
            models.Permission.name,
            models.RolesPermissionsMap.is_editable,
        ]
    ).select_from(
        models.PartnersRolesMap.join(
            models.RolesPermissionsMap,
            models.PartnersRolesMap.role_id == models.RolesPermissionsMap.role_id
        ).join(
            models.Permission,
            models.RolesPermissionsMap.permission_id == models.Permission.id
        ).join(
            models.UserRole,
            models.RolesPermissionsMap.role_id == models.UserRole.id
        )
    ).where(
        models.PartnersRolesMap.partner_id == partner_id
    )

    if skip_roles:
        q = q.where(
            ~models.UserRole.name.in_(skip_roles)
        )

    if role_prefix:
        q = q.where(
            models.UserRole.name.ilike(f'{role_prefix}%')
        )

    return q


async def get_agency_roles(agency_id: int) -> list[models.UserRole]:
    query = db.select(models.UserRole).select_from(
        models.UserRole.join(models.PartnersRolesMap)
    ).where(
        models.PartnersRolesMap.agency_id == agency_id
    )
    return await query.gino.all()


async def get_partner_roles(partner_id: int):
    query = db.select(models.UserRole).select_from(
        models.UserRole.join(models.PartnersRolesMap)
    ).where(
        models.PartnersRolesMap.partner_id == partner_id
    )
    return await query.gino.all()


async def get_role_permissions(role_id: int) -> list[models.Permission]:
    query = db.select(models.Permission).select_from(
        models.Permission.join(models.RolesPermissionsMap)
    ).where(
        models.RolesPermissionsMap.role_id == role_id
    )
    return await query.gino.all()
