from library.python.monlib.metric_registry import MetricRegistry
from smb.common.rmq.rpc.server import BaseRpcHandler
from crm.agency_cabinet.grants.common import structs
from crm.agency_cabinet.grants.server.src.procedures.role_manager import RoleManager, UserManager
from crm.agency_cabinet.grants.server.src.procedures.partner_manager import GetPartner, ListAvailablePartners, \
    GetPartnerID, ListPartners
from crm.agency_cabinet.grants.server.src.procedures.exceptions import (
    ConflictRoleException, NoSuchRoleException, NoSuchUserException,
    NoSuchPermissionException, UnsuitablePermissionException, NotEditablePermissionModificationException,
    NoPermission, PartnerNotFound, NoSuchOAuthTokenException, InactiveOAuthToken
)
from crm.agency_cabinet.grants.proto import access_level_pb2, request_pb2, role_pb2, common_pb2, grants_pb2, partners_pb2
from crm.agency_cabinet.common.blackbox import BlackboxClient


class Handler(BaseRpcHandler):
    _request_proto = request_pb2.RpcRequest  # TODO: use the same workaround for handling exceptions as for ord

    def __init__(self, blackbox_client: BlackboxClient = None, metric_registry: MetricRegistry = None):
        self.blackbox_client = blackbox_client
        self.metric_registry = metric_registry

    async def ping(self, _: common_pb2.Empty) -> common_pb2.PingOutput:
        return common_pb2.PingOutput(ping='pong')

    async def check_access_level(
        self, message: access_level_pb2.CheckAccessLevel
    ) -> access_level_pb2.CheckAccessLevelOutput:
        result = await RoleManager.check_access_level(structs.CheckAccessLevelRequest.from_proto(message))
        return access_level_pb2.CheckAccessLevelOutput(access_level=result.to_proto())

    async def get_access_scope(self, message: access_level_pb2.GetAccessScope) -> access_level_pb2.GetAccessScopeOutput:
        result = await RoleManager.get_access_scope(structs.GetAccessScopeRequest.from_proto(message))
        return access_level_pb2.GetAccessScopeOutput(result=result.to_proto())

    async def add_role(self, message: role_pb2.Role) -> role_pb2.AddRoleOutput:
        try:
            # TODO: remove
            await RoleManager.add_role(self.blackbox_client, structs.Role.from_proto(message))
        except ConflictRoleException:
            return role_pb2.AddRoleOutput(conflicting_role_exists=common_pb2.Empty())

        return role_pb2.AddRoleOutput(success=common_pb2.Empty())

    async def remove_role(self, message: role_pb2.Role) -> role_pb2.RemoveRoleOutput:
        try:
            # TODO: remove
            await RoleManager.remove_role(structs.Role.from_proto(message))
        except NoSuchRoleException:
            return role_pb2.RemoveRoleOutput(no_such_role=common_pb2.Empty())

        return role_pb2.RemoveRoleOutput(success=common_pb2.Empty())

    async def get_all_internal_roles(self, _: common_pb2.Empty) -> role_pb2.GetAllInternalRolesOutput:
        result = await RoleManager.get_all_internal_roles()
        return role_pb2.GetAllInternalRolesOutput(result=result.to_proto())

    async def add_internal_role(self, message: role_pb2.InternalRole) -> role_pb2.AddRoleOutput:
        try:
            await RoleManager.add_internal_role(self.blackbox_client, structs.InternalRole.from_proto(message))
        except ConflictRoleException:
            return role_pb2.AddRoleOutput(conflicting_role_exists=common_pb2.Empty())

        return role_pb2.AddRoleOutput(success=common_pb2.Empty())

    async def remove_internal_role(self, message: role_pb2.InternalRole) -> role_pb2.RemoveRoleOutput:
        try:
            await RoleManager.remove_internal_role(structs.InternalRole.from_proto(message))
        except NoSuchRoleException:
            return role_pb2.RemoveRoleOutput(no_such_role=common_pb2.Empty())

        return role_pb2.RemoveRoleOutput(success=common_pb2.Empty())

    async def get_users_permissions(
        self, message: grants_pb2.GetUsersPermissions
    ) -> grants_pb2.GetUsersPermissionsOutput:
        try:
            result = await UserManager.get_users_permissions(structs.GetUsersPermissionsRequest.from_proto(message))
            return grants_pb2.GetUsersPermissionsOutput(result=result.to_proto())
        except NoPermission as e:
            return grants_pb2.GetUsersPermissionsOutput(
                not_have_permission=common_pb2.ErrorMessageResponse(message=e.message)
            )

    async def edit_user(self, message: grants_pb2.EditUserRequest) -> grants_pb2.EditUserOutput:
        try:
            result = await UserManager.edit_user(structs.EditUserRequest.from_proto(message))
            return grants_pb2.EditUserOutput(result=result.to_proto())
        except NoSuchUserException as e:
            return grants_pb2.EditUserOutput(
                no_such_user=common_pb2.ErrorMessageResponse(message=e.message)
            )
        except NoSuchRoleException as e:
            return grants_pb2.EditUserOutput(
                no_such_role=common_pb2.ErrorMessageResponse(message=e.message)
            )
        except NoSuchPermissionException as e:
            return grants_pb2.EditUserOutput(
                no_such_permission=common_pb2.ErrorMessageResponse(message=e.message)
            )
        except UnsuitablePermissionException as e:
            return grants_pb2.EditUserOutput(
                unsuitable_permission=common_pb2.ErrorMessageResponse(message=e.message)
            )
        except NotEditablePermissionModificationException as e:
            return grants_pb2.EditUserOutput(
                not_editable_permission=common_pb2.ErrorMessageResponse(message=e.message)
            )
        except NoPermission as e:
            return grants_pb2.EditUserOutput(
                not_have_permission=common_pb2.ErrorMessageResponse(message=e.message)
            )

    async def get_suggested_users(self, message: grants_pb2.GetSuggestedUsers) -> grants_pb2.GetSuggestedUsersOutput:
        try:
            result = await UserManager.get_suggested_users(structs.GetSuggestedUsersRequest.from_proto(message))
            return grants_pb2.GetSuggestedUsersOutput(result=result.to_proto())
        except NoPermission as e:
            return grants_pb2.GetSuggestedUsersOutput(
                not_have_permission=common_pb2.ErrorMessageResponse(message=e.message)
            )

    async def list_roles(self, message: role_pb2.ListRolesRequest) -> role_pb2.ListRolesOutput:
        try:
            result = await UserManager.list_roles(structs.ListRolesRequest.from_proto(message))
            return role_pb2.ListRolesOutput(result=result.to_proto())
        except NoPermission as e:
            return role_pb2.ListRolesOutput(
                not_have_permission=common_pb2.ErrorMessageResponse(message=e.message)
            )

    async def list_user_roles(self, message: role_pb2.ListUserRolesRequest) -> role_pb2.ListUserRolesOutput:
        try:
            result = await UserManager.list_user_roles(structs.ListUserRolesRequest.from_proto(message))
            return role_pb2.ListUserRolesOutput(result=result.to_proto())
        except NoSuchUserException as e:
            return role_pb2.ListUserRolesOutput(
                no_such_user=common_pb2.ErrorMessageResponse(message=e.message)
            )
        except NoPermission as e:
            return role_pb2.ListUserRolesOutput(
                not_have_permission=common_pb2.ErrorMessageResponse(message=e.message)
            )

    async def check_permissions(self, message: grants_pb2.CheckPermissionsRequest) -> grants_pb2.CheckPermissionsOutput:
        try:
            result = await UserManager.check_permissions(structs.CheckPermissionsRequest.from_proto(message))
            return grants_pb2.CheckPermissionsOutput(result=result.to_proto())
        except NoSuchUserException as e:
            return grants_pb2.CheckPermissionsOutput(
                no_such_user=common_pb2.ErrorMessageResponse(message=e.message)
            )
        except NoSuchPermissionException as e:
            return grants_pb2.CheckPermissionsOutput(
                no_such_permissions=common_pb2.ErrorMessageResponse(message=e.message)
            )

    async def get_accessible_agencies(self, message: grants_pb2.GetAccessibleAgenciesRequest) -> grants_pb2.GetAccessibleAgenciesOutput:
        try:
            result = await UserManager.get_accessible_agencies(structs.GetAccessibleAgenciesRequest.from_proto(message))
            return grants_pb2.GetAccessibleAgenciesOutput(result=result.to_proto())
        except NoSuchUserException as e:
            return grants_pb2.GetAccessibleAgenciesOutput(
                no_such_user=common_pb2.ErrorMessageResponse(message=e.message)
            )

    async def list_user_permissions(self, message: grants_pb2.ListUserPermissionsRequest) -> grants_pb2.ListUserPermissionsOutput:
        try:
            result = await UserManager.list_user_permissions(
                structs.ListUserPermissionsRequest.from_proto(message))
            return grants_pb2.ListUserPermissionsOutput(result=result.to_proto())
        except NoSuchUserException as e:
            return grants_pb2.ListUserPermissionsOutput(
                no_such_user=common_pb2.ErrorMessageResponse(message=e.message)
            )
        except NoPermission as e:
            return grants_pb2.EditUserOutput(
                not_have_permission=common_pb2.ErrorMessageResponse(message=e.message)
            )

    async def list_available_partners(self, message: partners_pb2.ListAvailablePartnersInput) -> partners_pb2.ListAvailablePartnersOutput:
        try:
            result = await ListAvailablePartners()(structs.ListAvailablePartnersInput.from_proto(message))
            return partners_pb2.ListAvailablePartnersOutput(result=result.to_proto())
        except NoSuchUserException as e:
            return partners_pb2.ListAvailablePartnersOutput(
                no_such_user=common_pb2.ErrorMessageResponse(message=e.message)
            )

    async def get_partner(self, message: partners_pb2.GetPartnerInput) -> partners_pb2.GetPartnerOutput:
        try:
            result = await GetPartner()(structs.GetPartnerInput.from_proto(message))
            return partners_pb2.GetPartnerOutput(result=result.to_proto())
        except NoPermission as e:
            return partners_pb2.GetPartnerOutput(not_have_permission=common_pb2.ErrorMessageResponse(message=e.message))
        except PartnerNotFound as e:
            return partners_pb2.GetPartnerOutput(not_found=common_pb2.ErrorMessageResponse(message=e.message))

    async def get_partner_id(self, message: partners_pb2.GetPartnerIDInput) -> partners_pb2.GetPartnerIDOutput:
        try:
            result = await GetPartnerID()(structs.GetPartnerIDInput.from_proto(message))
            return partners_pb2.GetPartnerIDOutput(result=result.to_proto())
        except NoPermission as e:
            return partners_pb2.GetPartnerIDOutput(not_have_permission=common_pb2.ErrorMessageResponse(message=e.message))
        except PartnerNotFound as e:
            return partners_pb2.GetPartnerIDOutput(not_found=common_pb2.ErrorMessageResponse(message=e.message))

    async def check_oauth_permissions(self, message: grants_pb2.CheckOAuthPermissionsRequest) -> grants_pb2.CheckOAuthPermissionsOutput:
        try:
            result = await UserManager.check_oauth_permissions(
                structs.CheckOAuthPermissionsRequest.from_proto(message)
            )
            return grants_pb2.CheckOAuthPermissionsOutput(result=result.to_proto())
        except NoSuchOAuthTokenException as e:
            return grants_pb2.CheckOAuthPermissionsOutput(
                no_such_oauth_token=common_pb2.ErrorMessageResponse(message=e.message)
            )
        except InactiveOAuthToken as e:
            return grants_pb2.CheckOAuthPermissionsOutput(
                inactive_oauth_token=common_pb2.ErrorMessageResponse(message=e.message)
            )
        except NoSuchPermissionException as e:
            return grants_pb2.CheckOAuthPermissionsOutput(
                no_such_permissions=common_pb2.ErrorMessageResponse(message=e.message)
            )

    async def list_partners(self, message: partners_pb2.ListPartnersInput) -> partners_pb2.ListPartnersOutput:
        result = await ListPartners()(structs.ListPartnersInput.from_proto(message))
        return partners_pb2.ListPartnersOutput(result=result.to_proto())
