import logging
import typing
from dataclasses import dataclass
from sqlalchemy import and_
from sqlalchemy.engine.result import RowProxy
from crm.agency_cabinet.grants.common.consts.partner import PartnerType
from crm.agency_cabinet.grants.common import structs
from crm.agency_cabinet.grants.server.src.db import models, db
from crm.agency_cabinet.grants.server.src.procedures.exceptions import PartnerNotFound, NoPermission
from crm.agency_cabinet.grants.server.src.procedures.role_manager import UserManager

LOGGER = logging.getLogger('grants.partner_manager')


@dataclass
class ListPartners:
    async def __call__(self, request: structs.ListPartnersInput) -> structs.ListPartnersResponse:
        query = db.select(
            [
                models.Partner.id.label('partner_id'),
                models.Partner.external_id,
                models.Partner.type,
                models.Partner.name
            ]
        ).select_from(
            models.Partner
        )
        if request.type is not None:
            query = query.where(models.Partner.type == request.type.value)

        data = await query.gino.all()

        return structs.ListPartnersResponse(partners=[
            structs.Partner(
                partner_id=row.partner_id,
                type=PartnerType(row.type),
                external_id=row.external_id,
                name=row.name
            ) for row in data]
        )


@dataclass
class ListAvailablePartners:

    @staticmethod
    async def list_accessible_partners(yandex_uid) -> typing.List[RowProxy]:
        is_inner = await UserManager.check_for_inner_user(yandex_uid)
        if is_inner:
            return await db.select(
                [
                    models.Partner.id.label('partner_id'),
                    models.Partner.external_id,
                    models.Partner.type,
                    models.Partner.name
                ]
            ).select_from(
                models.Partner
            ).where(
                models.Partner.type == PartnerType.agency.value
            ).gino.all()

        user = await UserManager.get_user_by_uid(yandex_uid)

        return await db.select(
            [
                models.PartnersRolesMap.partner_id.label('partner_id'),
                models.Partner.external_id,
                models.Partner.type,
                models.Partner.name
            ]
        ).select_from(
            models.UsersRolesPermissionsMap.join(
                models.PartnersRolesMap,
                models.PartnersRolesMap.role_id == models.UsersRolesPermissionsMap.role_id
            ).join(
                models.Partner,
                models.Partner.id == models.PartnersRolesMap.partner_id
            )
        ).where(
            and_(
                models.UsersRolesPermissionsMap.user_id == user.id,
                models.UsersRolesPermissionsMap.permission_id.isnot(None)
            )
        ).gino.all()

    async def __call__(self, request: structs.ListAvailablePartnersInput) -> structs.ListAvailablePartnersResponse:
        data = await self.list_accessible_partners(request.yandex_uid)
        return structs.ListAvailablePartnersResponse(partners=[
            structs.Partner(
                partner_id=row.partner_id,
                type=PartnerType(row.type),
                external_id=row.external_id,
                name=row.name
            ) for row in data]
        )


@dataclass
class GetPartner:
    async def __call__(self, request: structs.GetPartnerInput) -> structs.Partner:
        partner: models.Partner = await models.Partner.query.where(
            models.Partner.id == request.partner_id
        ).gino.first()

        if partner is None:
            raise PartnerNotFound(f'Partner with id {request.partner_id} not found')

        accessible_partners = {
            partner.partner_id for partner in await ListAvailablePartners.list_accessible_partners(
                request.yandex_uid)}

        if request.partner_id not in accessible_partners:
            raise NoPermission(message=f'No permissions for partner with id {request.partner_id}')

        return structs.Partner(
            external_id=partner.external_id,
            type=PartnerType(partner.type),
            partner_id=partner.id,
            name=partner.name
        )


@dataclass
class GetPartnerID:
    async def __call__(self, request: structs.GetPartnerIDInput) -> structs.GetPartnerIDResponse:
        partner: models.Partner = await models.Partner.query.where(
            and_(
                models.Partner.external_id == request.external_id,
                models.Partner.type == request.type.value
            )
        ).gino.first()

        if partner is None:
            raise PartnerNotFound(f'Partner with external id {request.external_id} and type {request.type} not found')

        accessible_partners = {
            partner.external_id for partner in await ListAvailablePartners.list_accessible_partners(
                request.yandex_uid)}

        if request.external_id not in accessible_partners:
            raise NoPermission(message=f'No permissions for partner with external id {request.external_id}')

        return structs.GetPartnerIDResponse(
            partner_id=partner.id
        )
