import logging
import typing
from collections import defaultdict
from sqlalchemy import and_, or_

from crm.agency_cabinet.common.cache import TTLCache, cached
from crm.agency_cabinet.grants.common import structs
from crm.agency_cabinet.grants.server.src.db import models, db
from crm.agency_cabinet.grants.server.src.db.queries import build_select_permissions, build_select_available_roles
from crm.agency_cabinet.common.blackbox import BlackboxClient
from crm.agency_cabinet.common.consts.auth import UserType
from crm.agency_cabinet.grants.common.consts import Permissions, Roles
from crm.agency_cabinet.grants.server.src.procedures.exceptions import NoSuchRoleException, \
    ConflictRoleException, NoSuchUserException, NoSuchPermissionException, NoPermission, InactiveOAuthToken, \
    UnsuitablePermissionException, NotEditablePermissionModificationException, NoSuchOAuthTokenException,\
    NoSuchPartnerException
from crm.agency_cabinet.grants.common.consts.partner import PartnerType
LOGGER = logging.getLogger('grants.role_manager')


class RoleManager:
    # TODO: remove
    @staticmethod
    def _is_yandex(role: models.Role) -> bool:
        return role.agency_id is None and role.staff_login is not None

    @staticmethod
    async def check_access_level(request: structs.CheckAccessLevelRequest) -> structs.CheckAccessLevelResponse:
        role = await models.Role.query.where(
            models.Role.yandex_uid == request.yandex_uid,
        ).gino.first()

        if role and role.is_active and (role.agency_id == request.agency_id or RoleManager._is_yandex(role)):
            level = structs.AccessLevel.ALLOW
        else:
            level = structs.AccessLevel.DENY

        return structs.CheckAccessLevelResponse(level=level)

    @staticmethod
    async def get_access_scope(request: structs.GetAccessScopeRequest) -> structs.GetAccessScopeResponse:
        roles = await models.Role.query.where(
            and_(
                models.Role.yandex_uid == request.yandex_uid,
                models.Role.is_active.is_(True)
            )
        ).gino.all()

        if any(RoleManager._is_yandex(role) for role in roles):
            return structs.GetAccessScopeResponse(scope_type=structs.AccessScopeType.ALL, agencies=[])
        else:
            return structs.GetAccessScopeResponse(
                scope_type=structs.AccessScopeType.BY_ID,
                agencies=[role.agency_id for role in roles]
            )

    @staticmethod
    async def add_role(blackbox_client: BlackboxClient, request: structs.Role) -> None:
        yandex_uid = await blackbox_client.get_uid_by_email(request.email, real_ip=request.real_ip)

        role = await models.Role.query.where(
            and_(
                or_(
                    models.Role.email == request.email,
                    models.Role.yandex_uid == yandex_uid
                ),
                models.Role.agency_id == request.agency_id,
            )
        ).gino.first()

        if role:
            raise ConflictRoleException()

        await models.Role.create(
            yandex_uid=yandex_uid,
            agency_id=request.agency_id,
            email=request.email,
            is_main=False
        )

    @staticmethod
    async def remove_role(request: structs.Role) -> None:
        role = await models.Role.query.where(
            and_(
                models.Role.email == request.email,
                models.Role.agency_id == request.agency_id,
            )
        ).gino.first()

        if not role:
            raise NoSuchRoleException()

        await role.delete()

    @staticmethod
    async def get_all_internal_roles() -> structs.GetAllInternalRolesResponse:
        roles = await models.Role.query.where(
            models.Role.staff_login.isnot(None)
        ).gino.all()

        return structs.GetAllInternalRolesResponse(
            roles=[structs.InternalRole(
                staff_login=role.staff_login,
                email=role.email,
            ) for role in roles])

    @staticmethod
    async def add_internal_role(blackbox_client: BlackboxClient, request: structs.InternalRole) -> None:
        yandex_uid = await blackbox_client.get_uid_by_email(request.email, real_ip=request.real_ip)

        role = await models.Role.query.where(
            or_(
                models.Role.staff_login == request.staff_login,
                models.Role.email == request.email,
                models.Role.yandex_uid == yandex_uid,
            )
        ).gino.first()

        if role:
            LOGGER.error("Couldn't add internal role for %s, role already exists", request.staff_login)
            raise ConflictRoleException()

        await models.Role.create(
            yandex_uid=yandex_uid,
            agency_id=None,
            staff_login=request.staff_login,
            email=request.email,
            is_main=False
        )

    @staticmethod
    async def remove_internal_role(request: structs.InternalRole) -> None:
        role = await models.Role.query.where(
            and_(
                models.Role.staff_login == request.staff_login,
                models.Role.email == request.email
            )
        ).gino.first()

        if not role:
            LOGGER.error("Couldn't remove internal role for %s, role not found", request.staff_login)
            raise NoSuchRoleException()

        await role.delete()


class UserManager:
    @classmethod
    @cached(cache=TTLCache(ttl=10 * 60, maxsize=256))
    async def get_user_by_uid(cls, yandex_uid):
        user = await models.User.query.where(
            models.User.yandex_uid == yandex_uid
        ).gino.first()

        if user is None:
            LOGGER.error("Couldn't find user with yandex_uid = %s", yandex_uid)
            raise NoSuchUserException(f"Couldn't find user with yandex_uid = {yandex_uid}")
        return user

    @classmethod
    async def _assert_user_have_permissions(cls, partner_id: int, yandex_uid: int, permissions: list[str]):
        res = await cls.check_permissions(
            request=structs.CheckPermissionsRequest(
                partner_id=partner_id,
                yandex_uid=yandex_uid,
                permissions=[structs.Permission(name=name) for name in permissions]
            )
        )
        if not res.is_have_permissions:
            raise NoPermission(f'Missing permissions: {", ".join((e.name for e in res.missed_permissions))}')

    @classmethod
    async def get_users_permissions(cls, request: structs.GetUsersPermissionsRequest) -> structs.GetUsersPermissionsResponse:
        await cls.check_agency_or_partner_temp(request)
        await cls._assert_user_have_permissions(request.partner_id, request.current_yandex_uid, [Permissions.roles.value])

        role_ids = {
            row[0] for row in await db.select([
                db.func.distinct(models.UserRole.id)
            ]).select_from(
                models.UserRole.join(models.PartnersRolesMap)
            ).where(models.PartnersRolesMap.partner_id == request.partner_id).gino.all()}

        sub_q = db.select(
            [
                models.UsersRolesPermissionsMap.user_id,
                models.UserRole.id,
                models.UserRole.name,
                models.UserRole.display_name,
                db.func.array_agg(models.Permission.name).label('permissions')
            ]
        ).select_from(models.UsersRolesPermissionsMap.join(models.Permission).outerjoin(models.UserRole)).where(
            and_(
                models.UsersRolesPermissionsMap.role_id.in_(role_ids),
                models.UsersRolesPermissionsMap.is_active.is_(True)
            )
        ).group_by(models.UsersRolesPermissionsMap.user_id, models.UserRole.id, models.UserRole.name)
        user_roles_permissions = defaultdict(list)
        for row in await sub_q.gino.all():
            user_roles_permissions[row.user_id].append(
                structs.RolePermissions(
                    permissions=[structs.Permission(p) for p in sorted(row.permissions)],
                    role=structs.UserRole(role_id=row.id, name=row.name, display_name=row.display_name)
                )
            )

        users_query = models.User.query.where(
            and_(models.User.type == UserType.external.value, models.User.id.in_(user_roles_permissions.keys()))
        )
        if request.query:
            q = request.query.lower()
            users_query = users_query.where(or_(
                models.User.email.ilike(f'%{q}%'),
                models.User.display_name.ilike(f'%{q}%'),
                models.User.login.ilike(f'%{q}%'),
            )).order_by(models.User.display_name)
        users = await users_query.gino.all()
        return structs.GetUsersPermissionsResponse(
            users=[structs.UserPermissions(
                user=structs.User(
                    user_id=user.id,
                    avatar_id=user.avatar_id,
                    email=user.email,
                    login=user.login,
                    name=user.display_name,
                    current=user.yandex_uid == request.current_yandex_uid
                ),
                roles_permissions=user_roles_permissions[user.id]
            ) for user in users]
        )

    @classmethod
    async def edit_user(cls, request: structs.EditUserRequest):
        # логика аналогична list_roles, если нужна большая гранулярность, то:
        # TODO: get permissions list and check more granular
        await cls.check_agency_or_partner_temp(request)
        await cls._assert_user_have_permissions(request.partner_id, request.yandex_uid, [Permissions.roles.value])

        async with db.transaction(read_only=False, reuse=False):
            user = await models.User.query.where(
                models.User.id == request.user_id
            ).gino.first()

            if user is None:
                LOGGER.error("Couldn't find user with id equal to %s", str(request.user_id))
                raise NoSuchUserException(f"Couldn't find user with id equal to {request.user_id}")

            agency_base_role = await db.select([
                models.UserRole.id
            ]).select_from(
                models.UserRole.join(models.PartnersRolesMap).outerjoin(models.RolesPermissionsMap)
            ).where(
                and_(
                    models.PartnersRolesMap.partner_id == request.partner_id,
                    models.RolesPermissionsMap.permission_id.is_(None)
                )
            ).gino.first()

            agency_base_role_id = None
            if agency_base_role is None:
                LOGGER.warning("Couldn't find base role for partner_id = %s", request.partner_id)
            else:
                agency_base_role_id = agency_base_role.id

            agency_role_ids = await models.PartnersRolesMap.select('role_id').where(
                models.PartnersRolesMap.partner_id == request.partner_id
            ).gino.all()
            agency_role_ids = {role.role_id for role in agency_role_ids}
            agency_role_ids.discard(agency_base_role_id)

            request_role_ids = {role.role_id for role in request.roles}

            all_permissions = await models.Permission.query.gino.all()
            permission_name_to_id, permission_id_to_name = {}, {}
            for permission in all_permissions:
                permission_name_to_id[permission.name] = permission.id
                permission_id_to_name[permission.id] = permission.name

            for request_role in request.roles:
                if request_role.role_id not in agency_role_ids:
                    LOGGER.error(
                        "Couldn't find role with id equal to %s for partner_id=%s and user_id=%s",
                        str(request_role.role_id),
                        str(request.partner_id),
                        str(request.user_id),
                    )
                    raise NoSuchRoleException(
                        f"Couldn't find role with id equal to {request_role.role_id} for agency"
                    )

                db_role_permissions = await models.RolesPermissionsMap.query.where(
                    models.RolesPermissionsMap.role_id == request_role.role_id
                ).gino.all()
                db_role_permission_ids = {permission.permission_id for permission in db_role_permissions}

                request_role_permission_ids = set()
                for request_permission in request_role.permissions:
                    permission_id = permission_name_to_id.get(request_permission.name, None)

                    if permission_id is None:
                        LOGGER.error("Couldn't find permission with name '%s'", request_permission.name)
                        raise NoSuchPermissionException(
                            f"Couldn't find permission with name '{request_permission.name}'"
                        )
                    elif permission_id not in db_role_permission_ids:
                        LOGGER.error(
                            "Couldn't assign permission with id equal to %s for role_id = %s "
                            "to user with id equal to %s for agency with id equal to %s",
                            permission_id, request_role.role_id, request.user_id, request.partner_id
                        )
                        raise UnsuitablePermissionException(
                            f"Couldn't assign permission '{request_permission.name}' to user "
                            f"for role_id = {request_role.role_id}"
                        )
                    else:
                        request_role_permission_ids.add(permission_id)

                for permission in db_role_permissions:
                    if not permission.is_editable and permission.permission_id not in request_role_permission_ids:
                        LOGGER.error(
                            "Couldn't revoke permission with id equal to %s "
                            "for user with id equal to %s for agency with id equal to %s",
                            permission.permission_id, request.user_id, request.partner_id
                        )

                        raise NotEditablePermissionModificationException(
                            f"Couldn't revoke permission '{permission_id_to_name[permission.permission_id]}' of user "
                            f"for role_id = {request_role.role_id}"
                        )

                    permission_to_update = await models.UsersRolesPermissionsMap.query.where(
                        and_(
                            models.UsersRolesPermissionsMap.user_id == request.user_id,
                            models.UsersRolesPermissionsMap.role_id == request_role.role_id,
                            models.UsersRolesPermissionsMap.permission_id == permission.permission_id
                        )
                    ).gino.first()

                    if permission_to_update is None:
                        await models.UsersRolesPermissionsMap.create(
                            user_id=request.user_id,
                            role_id=request_role.role_id,
                            permission_id=permission.permission_id,
                            is_editable=permission.is_editable,
                            is_active=permission.permission_id in request_role_permission_ids
                        )
                    else:
                        await permission_to_update.update(
                            is_editable=permission.is_editable,
                            is_active=permission.permission_id in request_role_permission_ids
                        ).apply()

            roles_to_remove = {role for role in agency_role_ids if role not in request_role_ids}
            if agency_base_role_id is not None:
                roles_to_remove.add(agency_base_role_id)

            await models.UsersRolesPermissionsMap.delete.where(
                and_(
                    models.UsersRolesPermissionsMap.user_id == request.user_id,
                    models.UsersRolesPermissionsMap.role_id.in_(roles_to_remove),
                )
            ).gino.status(read_only=False, reuse=False)

            roles_to_remove.discard(agency_base_role_id)

            if agency_base_role_id is not None and (
                len(roles_to_remove) == len(agency_role_ids) and roles_to_remove == agency_role_ids
            ):
                await models.UsersRolesPermissionsMap.create(
                    user_id=request.user_id,
                    role_id=agency_base_role_id,
                    permission_id=None,
                    is_editable=False,
                    is_active=True
                )

            return structs.EditUserResponse(is_edited=True)

    @classmethod
    async def get_suggested_users(cls, request: structs.GetSuggestedUsersRequest) -> structs.GetSuggestedUsersResponse:
        await cls.check_agency_or_partner_temp(request)
        await cls._assert_user_have_permissions(request.partner_id, request.current_yandex_uid, [Permissions.roles.value])

        role_ids = {
            row[0] for row in await db.select([
                db.func.distinct(models.UserRole.id)
            ]).select_from(
                models.UserRole.join(models.PartnersRolesMap).outerjoin(models.RolesPermissionsMap)
            ).where(and_(
                models.PartnersRolesMap.partner_id == request.partner_id,
                models.RolesPermissionsMap.permission_id.is_(None)
            )).gino.all()
        }
        users_ids = {
            row[0] for row in await db.select(
                [
                    db.func.distinct(models.UsersRolesPermissionsMap.user_id)
                ]
            ).select_from(models.UsersRolesPermissionsMap).where(
                and_(
                    models.UsersRolesPermissionsMap.role_id.in_(role_ids),
                    models.UsersRolesPermissionsMap.permission_id.is_(None),
                )
            ).gino.all()
        }

        users_query = models.User.query.where(
            and_(models.User.type == UserType.external.value, models.User.id.in_(users_ids))
        ).order_by(
            models.User.id
        )
        if request.query:
            q = request.query.lower()
            users_query = users_query.where(or_(
                models.User.email.ilike(f'%{q}%'),
                models.User.display_name.ilike(f'%{q}%'),
                models.User.login.ilike(f'%{q}%'),
            )).order_by(models.User.display_name)
        users = await users_query.gino.all()

        return structs.GetSuggestedUsersResponse(
            users=[
                structs.User(
                    user_id=user.id,
                    avatar_id=user.avatar_id,
                    email=user.email,
                    login=user.login,
                    name=user.display_name,
                    current=False
                ) for user in users]
        )

    @classmethod
    async def _build_grouped_role_permissions(cls, partner_id) -> typing.Dict[int, typing.Dict]:
        # на фронте должна отсутствовать возможность редактировать/выдавать owner'ов, поэтому
        # не возвращаем их в list_roles
        # TODO?: owner'ов все еще можно редактировать через edit_user
        query = build_select_available_roles(partner_id, skip_roles=(Roles.owner.value, ))
        permissions = await query.gino.all()

        grouped_permissions = {}
        for permission in sorted(permissions, key=lambda p: (p.role_id, p.name)):
            if grouped_permissions.get(permission.role_id) is None:
                grouped_permissions[permission.role_id] = {
                    'role_id': permission.role_id,
                    'name': permission.role_name,
                    'display_name': permission.role_display_name,
                    'permissions': [{'name': permission.name, 'is_editable': permission.is_editable}]
                }
            else:
                grouped_permissions[permission.role_id]['permissions'].append(
                    {'name': permission.name, 'is_editable': permission.is_editable}
                )
        return grouped_permissions

    @classmethod
    async def list_roles(cls, request: structs.ListRolesRequest) -> structs.ListRolesResponse:
        # пользователь с доступ roles может редактировать любые роли, кроме owner'а
        # если понадобится более гранулярное разбиение (например, чтобы admin не мог редактировать admin'ов, то
        # в этом месте надо изменить проверку на получение списка ролей текущего пользователя)
        await cls.check_agency_or_partner_temp(request)
        await cls._assert_user_have_permissions(request.partner_id, request.yandex_uid, [Permissions.roles.value])

        grouped_permissions = await cls._build_grouped_role_permissions(request.partner_id)

        roles: list[structs.RolePermissions] = []
        for role_id, role in grouped_permissions.items():
            roles.append(
                structs.RolePermissions(
                    role=structs.UserRole(role_id=role['role_id'], name=role['name'], display_name=role['display_name']),
                    permissions=[
                        structs.Permission(
                            name=permission['name'],
                            is_editable=permission['is_editable']
                        )
                        for permission in role['permissions']
                    ]
                )
            )

        return structs.ListRolesResponse(roles=roles)

    @classmethod
    async def list_user_roles(cls, request: structs.ListUserRolesRequest) -> structs.ListUserRolesResponse:
        await cls.check_agency_or_partner_temp(request)
        is_inner = await UserManager.check_for_inner_user(request.yandex_uid)
        if is_inner:
            query = build_select_available_roles(request.partner_id, role_prefix=Roles.owner.value)
        else:
            user = await cls.get_user_by_uid(request.yandex_uid)
            query = build_select_permissions(request.partner_id, user.id, with_roles=True)

        permissions = await query.gino.all()

        grouped_permissions = {}
        for permission in sorted(permissions, key=lambda p: (p.role_id, p.name)):
            if grouped_permissions.get(permission.role_id) is None:
                grouped_permissions[permission.role_id] = {
                    'role_id': permission.role_id,
                    'name': permission.role_name,
                    'display_name': permission.role_display_name,
                    'permissions': [{'name': permission.name}]
                }
            else:
                grouped_permissions[permission.role_id]['permissions'].append({'name': permission.name})

        roles: list[structs.RolePermissions] = []
        for role_id, role in grouped_permissions.items():
            roles.append(
                structs.RolePermissions(
                    role=structs.UserRole(role_id=role['role_id'], name=role['name'], display_name=role['display_name']),
                    permissions=[
                        structs.Permission(name=permission['name'])
                        for permission in role['permissions']
                    ]
                )
            )

        if len(roles) == 0:
            raise NoPermission('Suggested user has no permissions')

        return structs.ListUserRolesResponse(roles=roles)

    @staticmethod
    async def _check_any_permissions(request: structs.CheckPermissionsRequest, user_id: int) -> structs.CheckPermissionsResponse:
        permissions_count = await db.select(
            [
                db.func.count(models.Permission.name.label('name')),
            ]
        ).select_from(
            models.UsersRolesPermissionsMap.join(
                models.Permission,
                models.UsersRolesPermissionsMap.permission_id == models.Permission.id
            ).join(
                models.PartnersRolesMap,
                models.PartnersRolesMap.role_id == models.UsersRolesPermissionsMap.role_id
            )
        ).where(
            and_(
                models.PartnersRolesMap.partner_id == request.partner_id,
                models.UsersRolesPermissionsMap.user_id == user_id,
                models.UsersRolesPermissionsMap.is_active.is_(True),
            )
        ).gino.scalar()
        return structs.CheckPermissionsResponse(
            is_have_permissions=permissions_count > 0,
            partner=structs.Partner.from_model(request.context.get('partner'))
        )

    @classmethod
    async def _check_if_permissions_exist(cls, request_permissions: typing.List[structs.Permission]):
        request_permissions_names = {p.name for p in request_permissions}
        existing_permissions = await db.select(
            [
                models.Permission.id,
                models.Permission.name
            ]
        ).select_from(models.Permission).where(models.Permission.name.in_(request_permissions_names)).gino.all()

        diff = set(request_permissions_names) - {p.name for p in existing_permissions}
        if diff:
            msg = ', '.join(diff)
            LOGGER.error("Couldn't find permissions: %s", msg)
            raise NoSuchPermissionException(
                f"Couldn't find permissions: {msg}"
            )
        return request_permissions_names, existing_permissions

    @classmethod
    async def _check_concrete_permissions(
        cls,
        request: structs.CheckPermissionsRequest,
        user_id: int,
        request_permissions: typing.List[structs.Permission]
    ) -> structs.CheckPermissionsResponse:
        request_permissions_names, permissions = await cls._check_if_permissions_exist(request_permissions)

        query = build_select_permissions(request.partner_id, user_id, permissions_ids=[p.id for p in permissions])
        user_permissions = {
            row.name for row in await query.gino.all()
        }

        diff = request_permissions_names - user_permissions
        if diff:
            return structs.CheckPermissionsResponse(
                is_have_permissions=False,
                missed_permissions=[structs.Permission(name=permission_name) for permission_name in diff],
                partner=structs.Partner.from_model(request.context.get('partner'))
            )
        else:
            return structs.CheckPermissionsResponse(
                is_have_permissions=True,
                partner=structs.Partner.from_model(request.context.get('partner'))
            )

    @classmethod
    async def _get_inner_user(cls, yandex_uid: int) -> models.Role:
        # TODO: delete
        return await models.Role.query.where(
            and_(
                models.Role.yandex_uid == yandex_uid,
                models.Role.staff_login.isnot(None),
                models.Role.agency_id.is_(None),
                models.Role.is_active.is_(True)
            )
        ).gino.first()

    @classmethod
    async def check_for_inner_user(cls, yandex_uid: int) -> bool:
        return await cls._get_inner_user(yandex_uid) is not None

    @classmethod
    async def check_agency_or_partner_temp(cls, request: typing.Union[structs.CheckPermissionsRequest,
                                                                      structs.CheckOAuthPermissionsRequest,
                                                                      structs.ListUserPermissionsRequest]):
        if getattr(request, 'agency_id', None) is not None and getattr(request, 'partner_id', None) is None:
            partner: models.Partner = await models.Partner.query.where(
                and_(
                    models.Partner.external_id == str(request.agency_id),
                    models.Partner.type == PartnerType.agency.value
                )
            ).gino.first()
            if partner is not None:
                request.partner_id = partner.id
                request.context['partner'] = partner
        elif getattr(request, 'partner_id', None) is not None:
            partner: models.Partner = await models.Partner.query.where(
                models.Partner.id == request.partner_id,
            ).gino.first()
            if partner is not None:
                request.context['partner'] = partner

        if request.context.get('partner') is None:
            raise NoSuchPartnerException()

    @classmethod
    async def check_permissions(cls, request: structs.CheckPermissionsRequest) -> structs.CheckPermissionsResponse:
        # TODO: temp, если есть внутренний пользователь (staff_login is not NULL and agency_id is NULL),
        #  то считаем то доступ есть везде
        await cls.check_agency_or_partner_temp(request)

        is_inner = await UserManager.check_for_inner_user(request.yandex_uid)
        if is_inner:
            return structs.CheckPermissionsResponse(
                is_have_permissions=True,
                partner=structs.Partner.from_model(request.context.get('partner'))
            )

        user = await cls.get_user_by_uid(request.yandex_uid)

        if request.permissions:
            return await cls._check_concrete_permissions(request, user.id, request.permissions)
        return await cls._check_any_permissions(request, user.id)

    @classmethod
    async def get_accessible_agencies(
        cls,
        request: structs.GetAccessibleAgenciesRequest
    ) -> structs.GetAccessibleAgenciesResponse:
        # TODO: temp, если есть внутренний пользователь (staff_login is not NULL and agency_id is NULL),
        #  то считаем то доступ ко всем агентствам
        is_inner = await UserManager.check_for_inner_user(request.yandex_uid)
        if is_inner:
            return structs.GetAccessibleAgenciesResponse(
                agencies=[
                    row.agency_id for row in await db.select(
                        [
                            db.func.distinct(models.PartnersRolesMap.agency_id).label('agency_id'),
                        ]
                    ).select_from(
                        models.PartnersRolesMap
                    ).where(
                        models.PartnersRolesMap.agency_id.isnot(None)
                    ).gino.all()
                ]
            )

        user = await cls.get_user_by_uid(request.yandex_uid)

        return structs.GetAccessibleAgenciesResponse(agencies=[
            row.agency_id for row in await db.select(
                [
                    models.PartnersRolesMap.agency_id.label('agency_id'),
                ]
            ).select_from(
                models.UsersRolesPermissionsMap.join(
                    models.PartnersRolesMap,
                    models.PartnersRolesMap.role_id == models.UsersRolesPermissionsMap.role_id
                )
            ).where(
                and_(
                    models.UsersRolesPermissionsMap.user_id == user.id,
                    models.UsersRolesPermissionsMap.permission_id.isnot(None)
                )
            ).gino.all()])

    @classmethod
    async def _get_user_permissions(cls, partner_id, user_id, prefix):
        query = build_select_permissions(partner_id, user_id, prefix=prefix)
        return await query.gino.all()

    @classmethod
    async def list_user_permissions(cls,
                                    request: structs.ListUserPermissionsRequest) -> structs.ListUserPermissionsResponse:
        await cls.check_agency_or_partner_temp(request)
        user = await cls.get_user_by_uid(request.yandex_uid)

        result = await cls._get_user_permissions(request.partner_id, user.id, prefix=request.permissions_prefix)

        return structs.ListUserPermissionsResponse(
            permissions=[structs.Permission(name=p.name) for p in result]
        )

    @classmethod
    async def check_oauth_permissions(
        cls, request: structs.CheckOAuthPermissionsRequest
    ) -> structs.CheckPermissionsResponse:
        app_client_id = await db.select(
            [
                models.AppOAuth.app_id,
                models.AppOAuth.is_active,
                models.AppOAuth.partner_id,
                db.func.array_agg(models.Permission.name).label('permissions'),
            ]
        ).select_from(
            models.AppOAuth.join(models.Partner).outerjoin(models.OAuthPermissionsMap).outerjoin(models.Permission)
        ).where(
            models.AppOAuth.app_id == request.app_client_id,
        ).group_by(
            models.AppOAuth.app_id, models.AppOAuth.is_active, models.AppOAuth.partner_id
        ).gino.first()

        if not app_client_id:
            raise NoSuchOAuthTokenException()

        if app_client_id.is_active is False:
            raise InactiveOAuthToken()

        request_permissions_names, _ = await cls._check_if_permissions_exist(request.permissions)

        diff = set(request_permissions_names) - set(app_client_id.permissions)

        request.partner_id = app_client_id.partner_id
        await cls.check_agency_or_partner_temp(request)

        if diff:
            return structs.CheckPermissionsResponse(
                is_have_permissions=False,
                missed_permissions=[structs.Permission(name=permission_name) for permission_name in diff],
                partner=structs.Partner.from_model(request.context.get('partner'))
            )
        else:
            return structs.CheckPermissionsResponse(
                is_have_permissions=True,
                partner=structs.Partner.from_model(request.context.get('partner'))
            )
