import typing
from datetime import datetime
from decimal import Decimal
from google.protobuf.message import Message
from crm.agency_cabinet.common.client import BaseClient, ProtocolError
from crm.agency_cabinet.ord.client.exceptions import PROTO_FIELD_TO_EXCEPTION
from crm.agency_cabinet.ord.common import QUEUE, structs, consts
from crm.agency_cabinet.ord.proto import common_pb2, request_pb2, reports_pb2, clients_pb2, import_data_pb2, \
    campaigns_pb2, acts_pb2, organizations_pb2, contracts_pb2, invites_pb2, client_rows_pb2


class OrdClient(BaseClient):
    queue = QUEUE

    async def ping(self) -> str:
        _, data = await self._send_message(
            request_pb2.RpcRequest(ping=common_pb2.Empty()),
            common_pb2.PingOutput
        )
        return str(data)

    @staticmethod
    def _process_exception(result_or_error: str, data: Message, result_field_name='result') -> Message:
        if result_or_error == result_field_name:
            return data
        exception_cls = PROTO_FIELD_TO_EXCEPTION.get(result_or_error)
        if exception_cls is None:
            raise ProtocolError('Unexpected response {}'.format(result_or_error))
        else:
            if data.DESCRIPTOR.name == 'Empty':  # TODO: use ErrorMessageResponse
                raise exception_cls()
            else:
                error_message = structs.ErrorMessageResponse.from_proto(data).message
                raise exception_cls(error_message)

    async def get_reports_info(
        self,
        agency_id: int,
        search_query: str = None,
        period_from: datetime.date = None,
        period_to: datetime.date = None,
        status: structs.ReportStatuses = None,
        sort=None,
        limit: int = None,
        offset: int = None,
    ) -> list[structs.ReportInfo]:

        if sort is None:
            sort = []
        request_pb = request_pb2.RpcRequest(
            get_reports_info=structs.GetReportsInfoRequest(
                agency_id=agency_id,
                search_query=search_query,
                period_from=period_from,
                period_to=period_to,
                status=status,
                sort=sort,
                limit=limit,
                offset=offset
            ).to_proto()
        )

        result, data = await self._send_message(
            request_pb,
            reports_pb2.GetReportsInfoOutput
        )

        if result == 'result':
            return structs.GetReportsInfoResponse.from_proto(data).reports

        raise ProtocolError('Unexpected response')

    async def get_detailed_report_info(self, agency_id: int, report_id: int) -> structs.ReportInfo:
        request_pb = request_pb2.RpcRequest(
            get_detailed_report_info=reports_pb2.GetDetailedReportInfo(
                agency_id=agency_id,
                report_id=report_id
            )
        )

        data = self._process_exception(*await self._send_message(
            request_pb,
            reports_pb2.GetDetailedReportInfoOutput
        ))

        return structs.ReportInfo.from_proto(data)

    async def get_report_clients_info(
        self,
        agency_id: int,
        report_id: int,
        is_valid: bool = None,
        limit: int = None,
        offset: int = None,
        search_query: str = None
    ) -> typing.List[structs.ClientInfo]:
        request = request_pb2.RpcRequest(
            get_report_clients_info=structs.GetReportClientsInfoInput(
                agency_id=agency_id,
                report_id=report_id,
                is_valid=is_valid,
                limit=limit,
                offset=offset,
                search_query=search_query
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request, clients_pb2.GetReportClientsInfoOutput
        ))

        return structs.ClientsInfoList.from_proto(data).clients

    async def send_report(self, agency_id: int, report_id: int):
        request = request_pb2.RpcRequest(
            send_report=structs.SendReportInput(
                agency_id=agency_id,
                report_id=report_id,
            ).to_proto()
        )

        self._process_exception(*await self._send_message(
            request, reports_pb2.SendReportOutput
        ))

    async def report_export(self, agency_id: int, report_id: int) -> structs.ReportExportResponse:
        request = request_pb2.RpcRequest(
            report_export=structs.ReportExportRequest(
                agency_id=agency_id,
                report_id=report_id
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request, reports_pb2.ReportExportOutput
        ))

        return structs.ReportExportResponse.from_proto(data)

    async def get_report_export_info(self, agency_id: int, report_id: int,
                                     report_export_id: int) -> structs.ReportExportResponse:

        request = request_pb2.RpcRequest(
            get_report_export_info=structs.ReportExportInfoRequest(
                agency_id=agency_id,
                report_id=report_id,
                report_export_id=report_export_id
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request, reports_pb2.ReportExportOutput
        ))

        return structs.ReportExportResponse.from_proto(data)

    async def get_report_url(self, agency_id: int, report_id: int, report_export_id: int) -> str:

        request = request_pb2.RpcRequest(
            get_report_url=structs.GetReportUrlRequest(
                agency_id=agency_id,
                report_id=report_id,
                report_export_id=report_export_id
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request, reports_pb2.GetReportUrlOutput
        ))

        return structs.GetReportUrlResponse.from_proto(data).report_url

    async def import_data(self, agency_id: int, report_id: int, filename: str, bucket: str, display_name: str = None) -> structs.TaskInfo:
        request = request_pb2.RpcRequest(
            import_data=structs.ImportDataInput(
                agency_id=agency_id,
                report_id=report_id,
                filename=filename,
                bucket=bucket,
                display_name=display_name
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request, import_data_pb2.ImportDataOutput
        ))

        return structs.TaskInfo.from_proto(data)

    async def get_lock_status(self, agency_id: int, report_id: int) -> structs.LockStatus:
        request = request_pb2.RpcRequest(
            get_lock_status=structs.GetLockStatusInput(
                agency_id=agency_id,
                report_id=report_id
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request, import_data_pb2.GetLockStatusOutput
        ))

        return structs.LockStatus.from_proto(data)

    async def delete_report(self,
                            agency_id: int,
                            report_id: int,
                            ):
        request = request_pb2.RpcRequest(
            delete_report=structs.DeleteReportRequest(
                agency_id=agency_id,
                report_id=report_id,
            ).to_proto()
        )

        self._process_exception(*await self._send_message(
            request, reports_pb2.DeleteReportOutput
        ))

    async def create_report(
        self,
        agency_id: int,
        period_from: datetime,
        reporter_type: consts.ReporterType,
    ) -> structs.ReportInfo:
        request_pb = request_pb2.RpcRequest(
            create_report=structs.CreateReportRequest(
                agency_id=agency_id,
                period_from=period_from,
                reporter_type=reporter_type,
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request_pb,
            reports_pb2.CreateReportOutput
        ))

        return structs.ReportInfo.from_proto(data)

    async def get_client_rows(self,
                              agency_id: int,
                              report_id: int,
                              client_id: int,
                              limit: int = None,
                              offset: int = None,
                              search_query: str = None,
                              is_valid: bool = None,
                              ) -> structs.ClientRowsList:
        request = request_pb2.RpcRequest(
            get_client_rows=structs.GetClientRowsInput(
                agency_id=agency_id,
                report_id=report_id,
                client_id=client_id,
                limit=limit,
                offset=offset,
                search_query=search_query,
                is_valid=is_valid,
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request, client_rows_pb2.GetClientRowsOutput
        ))

        return structs.ClientRowsList.from_proto(data)

    async def get_campaigns(self,
                            agency_id: int,
                            report_id: int,
                            client_id: int,
                            limit: int = None,
                            offset: int = None,
                            search_query: str = None
                            ) -> structs.CampaignList:

        request = request_pb2.RpcRequest(
            get_campaigns=structs.GetCampaignsInput(
                agency_id=agency_id,
                report_id=report_id,
                client_id=client_id,
                limit=limit,
                offset=offset,
                search_query=search_query
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request, campaigns_pb2.GetCampaignsOutput
        ))

        return structs.CampaignList.from_proto(data)

    async def get_acts(
        self,
        agency_id: int,
        report_id: int,
        search_query: str = None,
        limit: int = None,
        offset: int = None,
    ) -> structs.ActList:
        request = request_pb2.RpcRequest(
            get_acts=structs.GetActsInput(
                agency_id=agency_id,
                report_id=report_id,
                search_query=search_query,
                limit=limit,
                offset=offset,
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request, acts_pb2.GetActsOutput
        ))

        return structs.ActList.from_proto(data)

    async def edit_client_row(self,
                              agency_id: int,
                              report_id: int,
                              client_id: int,
                              row_id: int,
                              client_contract_id: typing.Optional[int] = None,
                              client_act_id: typing.Optional[int] = None,
                              ad_distributor_act_id: typing.Optional[int] = None,
                              campaign_eid: typing.Optional[str] = None,
                              campaign_name: typing.Optional[str] = None,
                              ):
        request = request_pb2.RpcRequest(
            edit_client_row=structs.EditClientRowInput(
                agency_id=agency_id,
                report_id=report_id,
                client_id=client_id,
                row_id=row_id,
                client_contract_id=client_contract_id,
                client_act_id=client_act_id,
                ad_distributor_act_id=ad_distributor_act_id,
                campaign_eid=campaign_eid,
                campaign_name=campaign_name,
            ).to_proto()
        )

        self._process_exception(*await self._send_message(
            request, client_rows_pb2.EditClientRowOutput
        ))

    async def get_client_short_info(self,
                                    agency_id: int,
                                    report_id: int,
                                    client_id: int,
                                    ) -> structs.ClientShortInfo:

        request_pb = request_pb2.RpcRequest(
            get_client_short_info=structs.ClientShortInfoInput(
                agency_id=agency_id,
                report_id=report_id,
                client_id=client_id
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request_pb, clients_pb2.ClientShortInfoOutput
        ))

        return structs.ClientShortInfo.from_proto(data)

    async def add_act(
        self,
        agency_id: int,
        report_id: int,
        act_eid: str,
        amount: typing.Optional[Decimal] = None,
        is_vat: typing.Optional[bool] = None,
    ) -> structs.Act:
        request = request_pb2.RpcRequest(
            add_act=structs.AddActInput(
                agency_id=agency_id,
                report_id=report_id,
                act_eid=act_eid,
                amount=amount,
                is_vat=is_vat,
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request, acts_pb2.AddActOutput
        ))

        return structs.Act.from_proto(data)

    async def edit_act(
        self,
        agency_id: int,
        report_id: int,
        act_id: int,
        act_eid: str,
        amount: typing.Optional[Decimal] = None,
        is_vat: typing.Optional[bool] = None,
    ) -> typing.NoReturn:
        request = request_pb2.RpcRequest(
            edit_act=structs.EditActInput(
                agency_id=agency_id,
                report_id=report_id,
                act_id=act_id,
                act_eid=act_eid,
                amount=amount,
                is_vat=is_vat,
            ).to_proto()
        )

        self._process_exception(*await self._send_message(
            request, acts_pb2.EditActOutput
        ))

    async def create_client(
        self,
        agency_id: int,
        report_id: int,
        client_id: str,
        login: typing.Optional[str] = None,
        name: typing.Optional[str] = None,
    ) -> structs.ClientInfo:
        request_pb = request_pb2.RpcRequest(
            create_client=structs.CreateClientInput(
                agency_id=agency_id,
                report_id=report_id,
                client_id=client_id,
                login=login,
                name=name,
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request_pb,
            clients_pb2.CreateClientOutput
        ))

        return structs.ClientInfo.from_proto(data)

    async def get_organizations(
        self,
        partner_id: int,
        limit: int = None,
        offset: int = None,
        search_query: str = None
    ) -> structs.OrganizationsList:
        request_pb = request_pb2.RpcRequest(
            get_organizations=structs.GetOrganizationsInput(
                partner_id=partner_id,
                limit=limit,
                offset=offset,
                search_query=search_query
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request_pb,
            organizations_pb2.GetOrganizationsOutput
        ))

        return structs.OrganizationsList.from_proto(data)

    async def get_contracts(
        self,
        agency_id: int,
        search_query: str = None,
        limit: int = None,
        offset: int = None,
    ) -> structs.ContractsList:
        request = request_pb2.RpcRequest(
            get_contracts=structs.GetContractsInput(
                search_query=search_query,
                limit=limit,
                offset=offset,
                agency_id=agency_id
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request, contracts_pb2.GetContractsOutput
        ))

        return structs.ContractsList.from_proto(data)

    async def get_invites(
        self,
        agency_id: int,
        limit: int = None,
        offset: int = None,
    ) -> structs.InviteList:
        request_pb = request_pb2.RpcRequest(
            get_invites=structs.GetInvitesInput(
                agency_id=agency_id,
                limit=limit,
                offset=offset,
            ).to_proto()
        )

        data = self._process_exception(*await self._send_message(
            request_pb,
            invites_pb2.GetInvitesOutput
        ))

        return structs.InviteList.from_proto(data)
