from typing import Optional, List
from decimal import Decimal
from dataclasses import dataclass

from crm.agency_cabinet.common.proto_utils import BaseStruct, safe_get_nullable_field, decimal_to_string
from crm.agency_cabinet.ord.proto import acts_pb2


@dataclass
class Act(BaseStruct):
    act_id: int
    act_eid: str
    amount: Optional[Decimal] = None
    is_vat: Optional[bool] = None

    @classmethod
    def from_proto(cls, message: acts_pb2.Act) -> 'Act':
        return cls(
            act_id=message.act_id,
            act_eid=message.act_eid,
            amount=safe_get_nullable_field(message, 'amount', Decimal),
            is_vat=safe_get_nullable_field(message, 'is_vat'),
        )

    def to_proto(self) -> acts_pb2.Act:
        return acts_pb2.Act(
            act_id=self.act_id,
            act_eid=self.act_eid,
            amount=decimal_to_string(self.amount),
            is_vat=self.is_vat,
        )


@dataclass
class AddActInput:
    agency_id: int
    report_id: int
    act_eid: str
    amount: Optional[Decimal] = None
    is_vat: Optional[bool] = None

    @classmethod
    def from_proto(cls, message: acts_pb2.AddActInput) -> 'AddActInput':
        return cls(
            agency_id=message.agency_id,
            report_id=message.report_id,
            act_eid=message.act_eid,
            amount=safe_get_nullable_field(message, 'amount', Decimal),
            is_vat=safe_get_nullable_field(message, 'is_vat'),
        )

    def to_proto(self) -> acts_pb2.AddActInput:
        return acts_pb2.AddActInput(
            agency_id=self.agency_id,
            report_id=self.report_id,
            act_eid=self.act_eid,
            amount=decimal_to_string(self.amount),
            is_vat=self.is_vat,
        )


@dataclass
class EditActInput:
    agency_id: int
    act_id: int
    report_id: int
    act_eid: Optional[str] = None
    amount: Optional[Decimal] = None
    is_vat: Optional[bool] = None

    @classmethod
    def from_proto(cls, message: acts_pb2.EditActInput) -> 'EditActInput':
        return cls(
            agency_id=message.agency_id,
            act_id=message.act_id,
            report_id=message.report_id,
            act_eid=safe_get_nullable_field(message, 'act_eid'),
            amount=safe_get_nullable_field(message, 'amount', Decimal),
            is_vat=safe_get_nullable_field(message, 'is_vat'),
        )

    def to_proto(self) -> acts_pb2.EditActInput:
        return acts_pb2.EditActInput(
            agency_id=self.agency_id,
            act_id=self.act_id,
            report_id=self.report_id,
            act_eid=self.act_eid,
            amount=decimal_to_string(self.amount),
            is_vat=self.is_vat,
        )


@dataclass
class GetActsInput(BaseStruct):
    agency_id: int
    report_id: int
    search_query: Optional[str] = None
    limit: Optional[int] = None
    offset: Optional[int] = None

    @classmethod
    def from_proto(cls, message: acts_pb2.GetActsInput) -> 'GetActsInput':
        return cls(
            search_query=message.search_query if message.HasField('search_query') else None,
            limit=message.limit if message.HasField('limit') else None,
            offset=message.offset if message.HasField('offset') else None,
            agency_id=message.agency_id,
            report_id=message.report_id,
        )

    def to_proto(self) -> acts_pb2.GetActsInput:
        return acts_pb2.GetActsInput(
            search_query=self.search_query,
            limit=self.limit,
            offset=self.offset,
            agency_id=self.agency_id,
            report_id=self.report_id,
        )


@dataclass
class ActList(BaseStruct):
    acts: List[Act]
    size: int

    @classmethod
    def from_proto(cls, message: acts_pb2.ActList) -> 'ActList':
        return cls(
            acts=[Act.from_proto(act) for act in message.acts],
            size=message.size,
        )

    def to_proto(self) -> acts_pb2.ActList:
        return acts_pb2.ActList(
            acts=[act.to_proto() for act in self.acts],
            size=self.size,
        )
