from typing import Optional, List
from dataclasses import dataclass

from crm.agency_cabinet.common.proto_utils import BaseStruct, safe_get_nullable_field
from crm.agency_cabinet.ord.proto import campaigns_pb2


@dataclass
class Campaign(BaseStruct):
    id: int
    campaign_eid: str
    name: Optional[str] = None
    creative_id: Optional[str] = None

    @classmethod
    def from_proto(cls, message: campaigns_pb2.Campaign) -> 'Campaign':
        return cls(
            id=message.id,
            campaign_eid=message.campaign_eid,
            name=safe_get_nullable_field(message, 'name'),
            creative_id=safe_get_nullable_field(message, 'creative_id'),
        )

    def to_proto(self) -> campaigns_pb2.Campaign:
        return campaigns_pb2.Campaign(
            id=self.id,
            campaign_eid=self.campaign_eid,
            name=self.name,
            creative_id=self.creative_id,
        )


@dataclass
class GetCampaignsInput(BaseStruct):
    agency_id: int
    report_id: int
    client_id: int
    limit: Optional[int] = None
    offset: Optional[int] = None
    search_query: Optional[str] = None

    @classmethod
    def from_proto(cls, message: campaigns_pb2.GetCampaignsInput) -> 'GetCampaignsInput':
        return cls(
            agency_id=message.agency_id,
            report_id=message.report_id,
            client_id=message.client_id,
            limit=safe_get_nullable_field(message, 'limit'),
            offset=safe_get_nullable_field(message, 'offset'),
            search_query=safe_get_nullable_field(message, 'search_query')
        )

    def to_proto(self) -> campaigns_pb2.GetCampaignsInput:
        return campaigns_pb2.GetCampaignsInput(
            agency_id=self.agency_id,
            report_id=self.report_id,
            client_id=self.client_id,
            limit=self.limit,
            offset=self.offset,
            search_query=self.search_query
        )


@dataclass
class CampaignList(BaseStruct):
    size: int
    campaigns: List[Campaign]

    @classmethod
    def from_proto(cls, message: campaigns_pb2.CampaignList) -> 'CampaignList':
        return cls(
            size=message.size,
            campaigns=[Campaign.from_proto(campaign) for campaign in message.campaigns],
        )

    def to_proto(self) -> campaigns_pb2.CampaignList:
        return campaigns_pb2.CampaignList(
            size=self.size,
            campaigns=[campaign.to_proto() for campaign in self.campaigns]
        )
