from typing import Optional, List
from decimal import Decimal
from dataclasses import dataclass

from smb.common.helpers import Converter

from crm.agency_cabinet.common.enum import BaseEnum
from crm.agency_cabinet.common.proto_utils import BaseStruct, safe_get_nullable_field, decimal_to_string
from crm.agency_cabinet.common.server.common.structs import SortTypes, sort_types_converter
from crm.agency_cabinet.ord.proto import client_rows_pb2
from crm.agency_cabinet.ord.common.structs import Act, Campaign, Contract, Organization


class ClientRowColumns(BaseEnum):
    is_valid = 'is_valid'


client_row_columns_converter = Converter(
    [
        (client_rows_pb2.ClientRowColumn.IS_VALID, ClientRowColumns.is_valid),
    ]
)


@dataclass
class ClientRowSort(BaseStruct):
    type: SortTypes
    column: ClientRowColumns

    @classmethod
    def from_proto(cls, message: client_rows_pb2.ClientRowSort) -> 'ClientRowSort':
        return cls(
            type=sort_types_converter.forward(message.type),
            column=client_row_columns_converter.forward(message.column)
        )

    def to_proto(self) -> client_rows_pb2.ClientRowSort:
        return client_rows_pb2.ClientRowSort(
            type=sort_types_converter.reversed(self.type),
            column=client_row_columns_converter.reversed(self.column)
        )


@dataclass
class ClientRow(BaseStruct):
    id: int
    suggested_amount: Optional[Decimal] = None
    campaign: Optional[Campaign] = None
    ad_distributor_organization: Optional[Organization] = None
    ad_distributor_partner_organization: Optional[Organization] = None
    partner_client_organization: Optional[Organization] = None
    advertiser_contractor_organization: Optional[Organization] = None
    advertiser_organization: Optional[Organization] = None
    ad_distributor_contract: Optional[Contract] = None
    ad_distributor_partner_contract: Optional[Contract] = None
    advertiser_contract: Optional[Contract] = None
    ad_distributor_act: Optional[Act] = None
    ad_distributor_partner_act: Optional[Act] = None

    @classmethod
    def from_proto(cls, message: client_rows_pb2.ClientRow) -> 'ClientRow':
        return cls(
            id=message.id,
            suggested_amount=safe_get_nullable_field(message, 'suggested_amount', Decimal),
            campaign=safe_get_nullable_field(message, 'campaign', Campaign.from_proto),
            ad_distributor_organization=safe_get_nullable_field(message, 'ad_distributor_organization', Organization.from_proto),
            ad_distributor_partner_organization=safe_get_nullable_field(message, 'ad_distributor_partner_organization', Organization.from_proto),
            partner_client_organization=safe_get_nullable_field(message, 'partner_client_organization', Organization.from_proto),
            advertiser_contractor_organization=safe_get_nullable_field(message, 'advertiser_contractor_organization', Organization.from_proto),
            advertiser_organization=safe_get_nullable_field(message, 'advertiser_organization', Organization.from_proto),
            ad_distributor_contract=safe_get_nullable_field(message, 'ad_distributor_contract', Contract.from_proto),
            ad_distributor_partner_contract=safe_get_nullable_field(message, 'ad_distributor_partner_contract', Contract.from_proto),
            advertiser_contract=safe_get_nullable_field(message, 'advertiser_contract', Contract.from_proto),
            ad_distributor_act=safe_get_nullable_field(message, 'ad_distributor_act', Act.from_proto),
            ad_distributor_partner_act=safe_get_nullable_field(message, 'ad_distributor_partner_act', Act.from_proto),
        )

    def to_proto(self) -> client_rows_pb2.ClientRow:
        return client_rows_pb2.ClientRow(
            id=self.id,
            suggested_amount=decimal_to_string(self.suggested_amount, '{:.2f}'),
            campaign=self.campaign.to_proto() if self.campaign else None,
            ad_distributor_organization=self.ad_distributor_organization.to_proto() if self.ad_distributor_organization else None,
            ad_distributor_partner_organization=self.ad_distributor_partner_organization.to_proto() if self.ad_distributor_partner_organization else None,
            partner_client_organization=self.partner_client_organization.to_proto() if self.partner_client_organization else None,
            advertiser_contractor_organization=self.advertiser_contractor_organization.to_proto() if self.advertiser_contractor_organization else None,
            advertiser_organization=self.advertiser_organization.to_proto() if self.advertiser_organization else None,
            ad_distributor_contract=self.ad_distributor_contract.to_proto() if self.ad_distributor_contract else None,
            ad_distributor_partner_contract=self.ad_distributor_partner_contract.to_proto() if self.ad_distributor_partner_contract else None,
            advertiser_contract=self.advertiser_contract.to_proto() if self.advertiser_contract else None,
            ad_distributor_act=self.ad_distributor_act.to_proto() if self.ad_distributor_act else None,
            ad_distributor_partner_act=self.ad_distributor_partner_act.to_proto() if self.ad_distributor_partner_act else None,
        )


@dataclass
class ClientRowsList(BaseStruct):
    size: int
    rows: List[ClientRow]

    @classmethod
    def from_proto(cls, message: client_rows_pb2.ClientRowsList) -> 'ClientRowsList':
        return cls(
            size=message.size,
            rows=[ClientRow.from_proto(row) for row in message.rows],
        )

    def to_proto(self) -> client_rows_pb2.ClientRowsList:
        return client_rows_pb2.ClientRowsList(
            size=self.size,
            rows=[row.to_proto() for row in self.rows]
        )


@dataclass
class GetClientRowsInput:
    agency_id: int
    report_id: int
    client_id: int
    limit: Optional[int] = None
    offset: Optional[int] = None
    search_query: Optional[str] = None
    is_valid: Optional[bool] = None

    @classmethod
    def from_proto(cls, message: client_rows_pb2.GetClientRowsInput) -> 'GetClientRowsInput':
        return cls(
            agency_id=message.agency_id,
            report_id=message.report_id,
            client_id=message.client_id,
            limit=safe_get_nullable_field(message, 'limit'),
            offset=safe_get_nullable_field(message, 'offset'),
            search_query=safe_get_nullable_field(message, 'search_query'),
            is_valid=safe_get_nullable_field(message, 'is_valid'),
        )

    def to_proto(self) -> client_rows_pb2.GetClientRowsInput:
        return client_rows_pb2.GetClientRowsInput(
            agency_id=self.agency_id,
            report_id=self.report_id,
            client_id=self.client_id,
            limit=self.limit,
            offset=self.offset,
            search_query=self.search_query,
            is_valid=self.is_valid,
        )


@dataclass
class EditClientRowInput:
    agency_id: int
    report_id: int
    client_id: int
    row_id: int
    client_contract_id: Optional[int] = None
    campaign_eid: Optional[str] = None
    campaign_name: Optional[str] = None
    client_act_id: Optional[int] = None
    ad_distributor_act_id: Optional[int] = None

    @classmethod
    def from_proto(cls, message: client_rows_pb2.EditClientRowInput) -> 'EditClientRowInput':
        return cls(
            agency_id=message.agency_id,
            report_id=message.report_id,
            row_id=message.row_id,
            client_id=message.client_id,
            client_act_id=safe_get_nullable_field(message, 'client_act_id'),
            ad_distributor_act_id=safe_get_nullable_field(message, 'ad_distributor_act_id'),
            client_contract_id=safe_get_nullable_field(message, 'client_contract_id'),
            campaign_eid=safe_get_nullable_field(message, 'campaign_eid'),
            campaign_name=safe_get_nullable_field(message, 'campaign_name'),
        )

    def to_proto(self) -> client_rows_pb2.EditClientRowInput:
        return client_rows_pb2.EditClientRowInput(
            agency_id=self.agency_id,
            report_id=self.report_id,
            client_id=self.client_id,
            row_id=self.row_id,
            client_act_id=self.client_act_id,
            ad_distributor_act_id=self.ad_distributor_act_id,
            client_contract_id=self.client_contract_id,
            campaign_eid=self.campaign_eid,
            campaign_name=self.campaign_name,
        )
