from typing import Optional, List
from decimal import Decimal
from dataclasses import dataclass

from crm.agency_cabinet.common.proto_utils import BaseStruct, safe_get_nullable_field, decimal_to_string
from crm.agency_cabinet.ord.proto import clients_pb2


@dataclass
class GetReportClientsInfoInput(BaseStruct):
    agency_id: int
    report_id: int
    is_valid: Optional[bool] = None
    limit: Optional[int] = None
    offset: Optional[int] = None
    search_query: Optional[str] = None

    @classmethod
    def from_proto(cls, message: clients_pb2.GetReportClientsInfoInput) -> 'GetReportClientsInfoInput':
        return cls(
            agency_id=message.agency_id,
            report_id=message.report_id,
            is_valid=safe_get_nullable_field(message, 'is_valid'),
            limit=safe_get_nullable_field(message, 'limit'),
            offset=safe_get_nullable_field(message, 'offset'),
            search_query=safe_get_nullable_field(message, 'search_query'),
        )

    def to_proto(self) -> clients_pb2.GetReportClientsInfoInput:
        return clients_pb2.GetReportClientsInfoInput(
            agency_id=self.agency_id,
            report_id=self.report_id,
            is_valid=self.is_valid,
            limit=self.limit,
            offset=self.offset,
            search_query=self.search_query
        )


@dataclass
class ClientInfo(BaseStruct):
    id: int
    client_id: str
    campaigns_count: int
    login: Optional[str] = None
    name: Optional[str] = None
    suggested_amount: Optional[Decimal] = None
    has_valid_ad_distributor: Optional[bool] = None
    has_valid_ad_distributor_partner: Optional[bool] = None
    has_valid_partner_client: Optional[bool] = None
    has_valid_advertiser: Optional[bool] = None
    has_valid_advertiser_contractor: Optional[bool] = None

    @classmethod
    def from_proto(cls, message: clients_pb2.ClientInfo) -> 'ClientInfo':
        return cls(
            id=message.id,
            client_id=message.client_id,
            login=safe_get_nullable_field(message, 'login'),
            name=safe_get_nullable_field(message, 'name'),
            suggested_amount=safe_get_nullable_field(message, 'suggested_amount', Decimal),
            campaigns_count=message.campaigns_count,
            has_valid_ad_distributor=safe_get_nullable_field(message, 'has_valid_ad_distributor'),
            has_valid_ad_distributor_partner=safe_get_nullable_field(message, 'has_valid_ad_distributor_partner'),
            has_valid_partner_client=safe_get_nullable_field(message, 'has_valid_partner_client'),
            has_valid_advertiser_contractor=safe_get_nullable_field(message, 'has_valid_advertiser_contractor'),
            has_valid_advertiser=safe_get_nullable_field(message, 'has_valid_advertiser'),

        )

    def to_proto(self) -> clients_pb2.GetReportClientsInfoInput:
        return clients_pb2.ClientInfo(
            id=self.id,
            client_id=self.client_id,
            login=self.login,
            name=self.name,
            suggested_amount=decimal_to_string(self.suggested_amount),
            campaigns_count=self.campaigns_count,
            has_valid_ad_distributor=self.has_valid_ad_distributor,
            has_valid_ad_distributor_partner=self.has_valid_ad_distributor_partner,
            has_valid_partner_client=self.has_valid_partner_client,
            has_valid_advertiser=self.has_valid_advertiser,
            has_valid_advertiser_contractor=self.has_valid_advertiser_contractor,
        )


@dataclass
class ClientsInfoList(BaseStruct):
    clients: List[ClientInfo]
    size: int

    @classmethod
    def from_proto(cls, message: clients_pb2.ClientsList) -> 'ClientsInfoList':
        return cls(
            clients=[ClientInfo.from_proto(c) for c in message.clients],
            size=message.size
        )

    def to_proto(self) -> clients_pb2.ClientsList:
        return clients_pb2.ClientsList(
            clients=[c.to_proto() for c in self.clients],
            size=self.size
        )


@dataclass
class ClientShortInfoInput(BaseStruct):
    agency_id: int
    report_id: int
    client_id: int

    @classmethod
    def from_proto(cls, message: clients_pb2.ClientShortInfoInput) -> 'ClientShortInfoInput':
        return cls(
            agency_id=message.agency_id,
            report_id=message.report_id,
            client_id=message.client_id,
        )

    def to_proto(self) -> clients_pb2.ClientShortInfoInput:
        return clients_pb2.ClientShortInfoInput(
            agency_id=self.agency_id,
            report_id=self.report_id,
            client_id=self.client_id,
        )


@dataclass
class ClientShortInfo(BaseStruct):
    id: int
    client_id: str
    login: Optional[str] = None
    name: Optional[str] = None

    @classmethod
    def from_proto(cls, message: clients_pb2.ClientShortInfo) -> 'ClientShortInfo':
        return cls(
            id=message.id,
            client_id=message.client_id,
            login=message.login,
            name=message.name,
        )

    def to_proto(self) -> clients_pb2.ClientShortInfo:
        return clients_pb2.ClientShortInfo(
            id=self.id,
            client_id=self.client_id,
            login=self.login,
            name=self.name,
        )


@dataclass
class CreateClientInput:
    agency_id: int
    report_id: int
    client_id: str
    login: Optional[str] = None
    name: Optional[str] = None

    @classmethod
    def from_proto(cls, message: clients_pb2.CreateClientInput) -> 'CreateClientInput':
        return cls(
            agency_id=message.agency_id,
            report_id=message.report_id,
            client_id=message.client_id,
            login=safe_get_nullable_field(message, 'login'),
            name=safe_get_nullable_field(message, 'name'),
        )

    def to_proto(self) -> clients_pb2.CreateClientInput:
        return clients_pb2.CreateClientInput(
            agency_id=self.agency_id,
            client_id=self.client_id,
            report_id=self.report_id,
            login=self.login,
            name=self.name,
        )
