import typing
from datetime import datetime
from decimal import Decimal
from dataclasses import dataclass
from smb.common.helpers import Converter
from smb.common.helpers import PbDatetimeConverter
from crm.agency_cabinet.ord.common.consts import ContractType, ContractActionType, ContractSubjectType
from crm.agency_cabinet.common.proto_utils import BaseStruct, decimal_to_string, safe_get_nullable_field
from crm.agency_cabinet.ord.proto import contracts_pb2
from .organizations import Organization


contract_type_converter = Converter(
    [
        (contracts_pb2.ContractType.CONTRACT, ContractType.contract),
        (contracts_pb2.ContractType.INTERMEDIARY_CONTRACT, ContractType.intermediary_contract),
        (contracts_pb2.ContractType.ADDITIONAL_AGREEMENT, ContractType.additional_agreement),
    ]
)

action_type_converter = Converter(
    [
        (contracts_pb2.ContractActionType.OTHER, ContractActionType.other),
        (contracts_pb2.ContractActionType.DISTRIBUTION, ContractActionType.distribution),
        (contracts_pb2.ContractActionType.CONCLUDE, ContractActionType.conclude),
        (contracts_pb2.ContractActionType.COMMERCIAL, ContractActionType.commercial),
    ]
)

subject_type_converter = Converter(
    [
        (contracts_pb2.ContractSubjectType.REPRESENTATION, ContractSubjectType.representation),
        (contracts_pb2.ContractSubjectType.OTHER_SUBJECT, ContractSubjectType.other),
        (contracts_pb2.ContractSubjectType.ORG_DISTRIBUTION, ContractSubjectType.org_distribution),
        (contracts_pb2.ContractSubjectType.MEDIATION, ContractSubjectType.mediation),
        (contracts_pb2.ContractSubjectType.DISTRIBUTION_SUBJECT, ContractSubjectType.distribution),
    ]
)


@dataclass
class Contract(BaseStruct):
    id: int
    contract_eid: typing.Optional[str] = None
    is_reg_report: typing.Optional[bool] = None
    type: typing.Optional[ContractType] = None
    action_type: typing.Optional[ContractActionType] = None
    subject_type: typing.Optional[ContractSubjectType] = None
    date: typing.Optional[datetime] = None
    amount: typing.Optional[Decimal] = None
    is_vat: typing.Optional[bool] = None

    client_organization: typing.Optional[Organization] = None
    contractor_organization: typing.Optional[Organization] = None

    @classmethod
    def from_proto(cls, message: contracts_pb2.Contract) -> 'Contract':
        return cls(
            id=message.id,
            contract_eid=safe_get_nullable_field(message, 'contract_eid'),
            is_reg_report=safe_get_nullable_field(message, 'is_reg_report'),
            type=safe_get_nullable_field(message, 'type', contract_type_converter.forward),
            action_type=safe_get_nullable_field(message, 'action_type', action_type_converter.forward),
            subject_type=safe_get_nullable_field(message, 'subject_type', subject_type_converter.forward),
            date=safe_get_nullable_field(message, 'date', PbDatetimeConverter().to_datetime),
            amount=safe_get_nullable_field(message, 'amount', Decimal),
            is_vat=safe_get_nullable_field(message, 'is_vat'),
            client_organization=safe_get_nullable_field(message, 'client_organization', Organization.from_proto),
            contractor_organization=safe_get_nullable_field(message, 'contractor_organization', Organization.from_proto)
        )

    def to_proto(self) -> contracts_pb2.Contract:
        return contracts_pb2.Contract(
            id=self.id,
            contract_eid=self.contract_eid,
            is_reg_report=self.is_reg_report,
            type=contract_type_converter.reversed(self.type) if self.type else None,
            action_type=action_type_converter.reversed(self.action_type) if self.action_type else None,
            subject_type=subject_type_converter.reversed(self.subject_type) if self.subject_type else None,
            date=PbDatetimeConverter().from_datetime(self.date) if self.date else None,
            amount=decimal_to_string(self.amount, '{:.2f}'),
            is_vat=self.is_vat,
            client_organization=self.client_organization.to_proto() if self.client_organization else None,
            contractor_organization=self.contractor_organization.to_proto() if self.contractor_organization else None
        )


@dataclass
class ContractsList(BaseStruct):
    size: int
    contracts: typing.List[Contract]

    @classmethod
    def from_proto(cls, message: contracts_pb2.ContractsList) -> 'ContractsList':
        return cls(
            contracts=[Contract.from_proto(c) for c in message.contracts],
            size=message.size
        )

    def to_proto(self) -> contracts_pb2.ContractsList:
        return contracts_pb2.ContractsList(
            contracts=[c.to_proto() for c in self.contracts],
            size=self.size
        )


@dataclass
class GetContractsInput(BaseStruct):
    agency_id: int
    search_query: typing.Optional[str] = None
    limit: typing.Optional[int] = None
    offset: typing.Optional[int] = None

    @classmethod
    def from_proto(cls, message: contracts_pb2.GetContractsInput) -> 'GetContractsInput':
        return cls(
            search_query=message.search_query if message.HasField('search_query') else None,
            limit=message.limit if message.HasField('limit') else None,
            offset=message.offset if message.HasField('offset') else None,
            agency_id=message.agency_id,
        )

    def to_proto(self) -> contracts_pb2.GetContractsInput:
        return contracts_pb2.GetContractsInput(
            search_query=self.search_query,
            limit=self.limit,
            offset=self.offset,
            agency_id=self.agency_id,
        )
