from typing import Optional, List
from dataclasses import dataclass
from datetime import datetime

from crm.agency_cabinet.common.proto_utils import BaseStruct, timestamp_or_none, safe_get_nullable_field
from crm.agency_cabinet.ord.proto import invites_pb2


@dataclass
class Invite(BaseStruct):
    invite_id: int
    email: str
    status: str
    created_at: datetime

    @classmethod
    def from_proto(cls, message: invites_pb2.Invite) -> 'Invite':
        return cls(
            invite_id=message.invite_id,
            email=message.email,
            status=message.status,
            created_at=message.created_at.ToDatetime(),
        )

    def to_proto(self) -> invites_pb2.Invite:
        return invites_pb2.Invite(
            invite_id=self.invite_id,
            email=self.email,
            status=self.status,
            created_at=timestamp_or_none(self.created_at),
        )


@dataclass
class InviteList(BaseStruct):
    invites: List[Invite]
    size: int

    @classmethod
    def from_proto(cls, message: invites_pb2.InviteList) -> 'InviteList':
        return cls(
            invites=[Invite.from_proto(invite) for invite in message.invites],
            size=message.size,
        )

    def to_proto(self) -> invites_pb2.InviteList:
        return invites_pb2.InviteList(
            invites=[invite.to_proto() for invite in self.invites],
            size=self.size,
        )


@dataclass
class GetInvitesInput(BaseStruct):
    agency_id: int
    limit: Optional[int] = None
    offset: Optional[int] = None

    @classmethod
    def from_proto(cls, message: invites_pb2.GetInvitesInput) -> 'GetInvitesInput':
        return cls(
            agency_id=message.agency_id,
            limit=safe_get_nullable_field(message, 'limit'),
            offset=safe_get_nullable_field(message, 'offset'),
        )

    def to_proto(self) -> invites_pb2.GetInvitesInput:
        return invites_pb2.GetInvitesInput(
            agency_id=self.agency_id,
            limit=self.limit if self.limit else None,
            offset=self.offset if self.offset else None,
        )
