from typing import Optional, List
from dataclasses import dataclass

from smb.common.helpers import Converter

from crm.agency_cabinet.common.proto_utils import BaseStruct, safe_get_nullable_field
from crm.agency_cabinet.ord.proto import organizations_pb2
from crm.agency_cabinet.ord.common.consts import OrganizationType


organization_type_converter = Converter(
    [
        (organizations_pb2.OrganizationType.FFL, OrganizationType.ffl),
        (organizations_pb2.OrganizationType.FUL, OrganizationType.ful),
        (organizations_pb2.OrganizationType.IP, OrganizationType.ip),
        (organizations_pb2.OrganizationType.FL, OrganizationType.fl),
        (organizations_pb2.OrganizationType.UL, OrganizationType.ul),
    ]
)


@dataclass
class GetOrganizationsInput(BaseStruct):
    partner_id: int
    limit: Optional[int] = None
    offset: Optional[int] = None
    search_query: Optional[str] = None

    @classmethod
    def from_proto(cls, message: organizations_pb2.GetOrganizationsInput) -> 'GetOrganizationsInput':
        return cls(
            partner_id=message.partner_id,
            limit=safe_get_nullable_field(message, 'limit'),
            offset=safe_get_nullable_field(message, 'offset'),
            search_query=safe_get_nullable_field(message, 'search_query')
        )

    def to_proto(self) -> organizations_pb2.GetOrganizationsInput:
        return organizations_pb2.GetOrganizationsInput(
            partner_id=self.partner_id,
            limit=self.limit,
            offset=self.offset,
            search_query=self.search_query
        )


@dataclass
class Organization(BaseStruct):
    id: int
    type: Optional[OrganizationType] = None
    name: Optional[str] = None
    inn: Optional[str] = None
    is_rr: Optional[bool] = None
    is_ors: Optional[bool] = None
    mobile_phone: Optional[str] = None
    epay_number: Optional[str] = None
    reg_number: Optional[str] = None
    alter_inn: Optional[str] = None
    oksm_number: Optional[str] = None
    rs_url: Optional[str] = None

    @classmethod
    def from_proto(cls, message: organizations_pb2.Organization) -> 'Organization':
        return cls(
            id=message.id,
            type=safe_get_nullable_field(message, 'type', organization_type_converter.forward),
            name=safe_get_nullable_field(message, 'name'),
            inn=safe_get_nullable_field(message, 'inn'),
            is_rr=safe_get_nullable_field(message, 'is_rr'),
            is_ors=safe_get_nullable_field(message, 'is_ors'),
            mobile_phone=safe_get_nullable_field(message, 'mobile_phone'),
            epay_number=safe_get_nullable_field(message, 'epay_number'),
            reg_number=safe_get_nullable_field(message, 'reg_number'),
            alter_inn=safe_get_nullable_field(message, 'alter_inn'),
            oksm_number=safe_get_nullable_field(message, 'oksm_number'),
            rs_url=safe_get_nullable_field(message, 'rs_url'),
        )

    def to_proto(self) -> organizations_pb2.Organization:
        return organizations_pb2.Organization(
            id=self.id,
            type=organization_type_converter.reversed(self.type) if self.type else None,
            name=self.name,
            inn=self.inn,
            is_rr=self.is_rr,
            is_ors=self.is_ors,
            mobile_phone=self.mobile_phone,
            epay_number=self.epay_number,
            reg_number=self.reg_number,
            alter_inn=self.alter_inn,
            oksm_number=self.oksm_number,
            rs_url=self.rs_url,
        )


@dataclass
class OrganizationsList(BaseStruct):
    size: int
    organizations: List[Organization]

    @classmethod
    def from_proto(cls, message: organizations_pb2.OrganizationsList) -> 'OrganizationsList':
        return cls(
            size=message.size,
            organizations=[Organization.from_proto(org) for org in message.organizations],
        )

    def to_proto(self) -> organizations_pb2.OrganizationsList:
        return organizations_pb2.OrganizationsList(
            size=self.size,
            organizations=[org.to_proto() for org in self.organizations]
        )
