from crm.agency_cabinet.common.enum import BaseEnum
from dataclasses import dataclass
from datetime import datetime
from typing import Optional

from crm.agency_cabinet.common.proto_utils import BaseStruct, safe_get_nullable_field
from crm.agency_cabinet.common.server.common.structs import SortTypes, TaskStatuses, sort_types_converter, \
    task_status_converter
from crm.agency_cabinet.ord.proto import reports_pb2
from smb.common.helpers import Converter, PbDatetimeConverter
from crm.agency_cabinet.ord.common.consts import ReportStatuses, ReporterType


report_status_converter = Converter(
    [
        (reports_pb2.ReportStatus.DRAFT, ReportStatuses.draft),
        (reports_pb2.ReportStatus.SENT, ReportStatuses.sent),
        (reports_pb2.ReportStatus.ALL, ReportStatuses.all)
    ]
)

reporter_type_converter = Converter(
    [
        (reports_pb2.ReporterType.PARTNER, ReporterType.partner),
        (reports_pb2.ReporterType.AD_DISTRIBUTOR, ReporterType.ad_distributor),
        (reports_pb2.ReporterType.ADVERTISER, ReporterType.advertiser)
    ]
)


class ReportColumns(BaseEnum):
    status = 'status'
    sending_date = 'sending_date'
    clients_count = 'clients_count'


report_columns_converter = Converter(
    [
        (reports_pb2.ReportColumn.STATUS, ReportColumns.status),
        (reports_pb2.ReportColumn.SENDING_DATE, ReportColumns.sending_date),
        (reports_pb2.ReportColumn.CLIENTS_COUNT, ReportColumns.clients_count),
    ]
)


@dataclass
class ReportSort(BaseStruct):
    type: SortTypes
    column: ReportColumns

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportSort) -> 'ReportSort':
        return cls(
            type=sort_types_converter.forward(message.type),
            column=report_columns_converter.forward(message.column)
        )

    def to_proto(self) -> 'ReportSort':
        return reports_pb2.ReportSort(
            type=sort_types_converter.reversed(self.type),
            column=report_columns_converter.reversed(self.column)
        )


@dataclass
class GetReportsInfoRequest(BaseStruct):
    agency_id: int
    sort: list[ReportSort]
    period_from: Optional[datetime] = None
    period_to: Optional[datetime] = None
    search_query: Optional[str] = None
    status: Optional[ReportStatuses] = ReportStatuses.all
    limit: Optional[int] = None
    offset: Optional[int] = None

    @classmethod
    def from_proto(cls, message: reports_pb2.GetReportsInfo) -> 'GetReportsInfoRequest':
        return cls(
            agency_id=message.agency_id,
            search_query=safe_get_nullable_field(message, 'search_query'),
            period_from=safe_get_nullable_field(message, 'period_from', PbDatetimeConverter().to_datetime),
            period_to=safe_get_nullable_field(message, 'period_to', PbDatetimeConverter().to_datetime),
            status=safe_get_nullable_field(message, 'status', report_status_converter.forward),
            sort=[ReportSort.from_proto(s) for s in message.sort] if message.sort else [],
            limit=safe_get_nullable_field(message, 'limit'),
            offset=safe_get_nullable_field(message, 'offset'),
        )

    def to_proto(self) -> reports_pb2.GetReportsInfo:
        return reports_pb2.GetReportsInfo(
            agency_id=self.agency_id,
            search_query=self.search_query,
            period_from=PbDatetimeConverter().from_datetime(self.period_from) if self.period_from else None,
            period_to=PbDatetimeConverter().from_datetime(self.period_to) if self.period_to else None,
            status=report_status_converter.reversed(self.status) if self.status else None,
            sort=[s.to_proto() for s in self.sort],
            limit=self.limit,
            offset=self.offset
        )


@dataclass
class ReportSettings(BaseStruct):
    name: str
    display_name: str
    allow_create_ad_distributor_acts: bool
    allow_create_clients: bool
    allow_create_campaigns: bool
    allow_edit_report: bool

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportSettings) -> 'ReportSettings':
        return cls(
            name=message.name,
            display_name=message.display_name,
            allow_create_ad_distributor_acts=message.allow_create_ad_distributor_acts,
            allow_create_clients=message.allow_create_clients,
            allow_create_campaigns=message.allow_create_campaigns,
            allow_edit_report=message.allow_edit_report,
        )

    def to_proto(self) -> reports_pb2.ReportSettings:
        return reports_pb2.ReportSettings(
            name=self.name,
            display_name=self.display_name,
            allow_create_ad_distributor_acts=self.allow_create_ad_distributor_acts,
            allow_create_clients=self.allow_create_clients,
            allow_create_campaigns=self.allow_create_campaigns,
            allow_edit_report=self.allow_edit_report,
        )


@dataclass
class ReportInfo(BaseStruct):
    report_id: int
    status: ReportStatuses
    period_from: Optional[datetime]
    reporter_type: ReporterType
    clients_count: int
    campaigns_count: int
    settings: ReportSettings
    sending_date: Optional[datetime] = None

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportInfo) -> 'ReportInfo':
        return cls(
            report_id=message.report_id,
            status=report_status_converter.forward(message.status),
            sending_date=PbDatetimeConverter().to_datetime(message.sending_date) if message.HasField('sending_date') else None,
            period_from=PbDatetimeConverter().to_datetime(message.period_from) if message.HasField('period_from') else None,
            reporter_type=reporter_type_converter.forward(message.reporter_type),
            clients_count=message.clients_count,
            campaigns_count=message.campaigns_count,
            settings=ReportSettings.from_proto(message.settings)
        )

    def to_proto(self) -> reports_pb2.ReportInfo:
        return reports_pb2.ReportInfo(
            report_id=self.report_id,
            status=report_status_converter.reversed(self.status),
            sending_date=PbDatetimeConverter().from_datetime(self.sending_date) if self.sending_date else None,
            period_from=PbDatetimeConverter().from_datetime(self.period_from) if self.period_from else None,
            reporter_type=reporter_type_converter.reversed(self.reporter_type),
            clients_count=self.clients_count,
            campaigns_count=self.clients_count,
            settings=self.settings.to_proto()
        )


@dataclass
class GetReportsInfoResponse(BaseStruct):
    reports: list[ReportInfo]

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportsInfoList) -> 'GetReportsInfoResponse':
        return cls(reports=[ReportInfo.from_proto(report) for report in message.reports])

    def to_proto(self) -> reports_pb2.ReportsInfoList:
        return reports_pb2.ReportsInfoList(reports=[report.to_proto() for report in self.reports])


@dataclass
class GetDetailedReportInfoRequest(BaseStruct):
    agency_id: int
    report_id: int

    @classmethod
    def from_proto(cls, message: reports_pb2.GetDetailedReportInfo) -> 'GetDetailedReportInfoRequest':
        return cls(agency_id=message.agency_id, report_id=message.report_id)

    def to_proto(self) -> reports_pb2.GetDetailedReportInfo:
        return reports_pb2.GetDetailedReportInfo(agency_id=self.agency_id, report_id=self.report_id)


@dataclass
class SendReportInput(BaseStruct):
    agency_id: int
    report_id: int

    @classmethod
    def from_proto(cls, message: reports_pb2.SendReportInput) -> 'SendReportInput':
        return cls(agency_id=message.agency_id, report_id=message.report_id)

    def to_proto(self) -> reports_pb2.SendReportInput:
        return reports_pb2.SendReportInput(agency_id=self.agency_id, report_id=self.report_id)


@dataclass
class ReportExportRequest(BaseStruct):
    agency_id: int
    report_id: int

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportExportInput) -> 'ReportExportRequest':
        return cls(
            agency_id=message.agency_id,
            report_id=message.report_id,
        )

    def to_proto(self) -> reports_pb2.ReportExportInput:
        return reports_pb2.ReportExportInput(
            agency_id=self.agency_id,
            report_id=self.report_id,
        )


@dataclass
class ReportExportInfoRequest(BaseStruct):
    agency_id: int
    report_id: int
    report_export_id: int

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportExportInfoInput) -> 'ReportExportInfoRequest':
        return cls(
            agency_id=message.agency_id,
            report_id=message.report_id,
            report_export_id=message.report_export_id
        )

    def to_proto(self) -> reports_pb2.ReportExportInfoInput:
        return reports_pb2.ReportExportInfoInput(
            agency_id=self.agency_id,
            report_id=self.report_id,
            report_export_id=self.report_export_id,
        )


@dataclass
class ReportExportResponse:
    report_export_id: int
    status: TaskStatuses

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportExport) -> 'ReportExportResponse':
        return cls(
            report_export_id=message.report_export_id,
            status=task_status_converter.forward(message.status)
        )

    def to_proto(self) -> reports_pb2.ReportExport:
        return reports_pb2.ReportExport(
            report_export_id=self.report_export_id,
            status=task_status_converter.reversed(self.status)
        )


@dataclass
class GetReportUrlRequest(BaseStruct):
    agency_id: int
    report_id: int
    report_export_id: int

    @classmethod
    def from_proto(cls, message: reports_pb2.GetReportUrlInput) -> 'GetReportUrlRequest':
        return cls(
            agency_id=message.agency_id,
            report_id=message.report_id,
            report_export_id=message.report_export_id
        )

    def to_proto(self) -> reports_pb2.GetReportUrlInput:
        return reports_pb2.GetReportUrlInput(
            agency_id=self.agency_id,
            report_id=self.report_id,
            report_export_id=self.report_export_id,
        )


@dataclass
class GetReportUrlResponse:
    report_url: str

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportUrl) -> 'GetReportUrlResponse':
        return cls(report_url=message.url)

    def to_proto(self) -> reports_pb2.ReportUrl:
        return reports_pb2.ReportUrl(url=self.report_url)


@dataclass()
class DeleteReportRequest:
    agency_id: int
    report_id: int

    @classmethod
    def from_proto(cls, message: reports_pb2.DeleteReportInput) -> 'DeleteReportRequest':
        return cls(
            agency_id=message.agency_id,
            report_id=message.report_id,
        )

    def to_proto(self) -> reports_pb2.DeleteReportInput:
        return reports_pb2.DeleteReportInput(
            agency_id=self.agency_id,
            report_id=self.report_id,
        )


@dataclass
class CreateReportRequest:
    agency_id: int
    period_from: datetime
    reporter_type: ReporterType

    @classmethod
    def from_proto(cls, message: reports_pb2.CreateReportInput) -> 'CreateReportRequest':
        return cls(
            agency_id=message.agency_id,
            period_from=PbDatetimeConverter().to_datetime(message.period_from),
            reporter_type=reporter_type_converter.forward(message.reporter_type),
        )

    def to_proto(self) -> reports_pb2.CreateReportInput:
        return reports_pb2.CreateReportInput(
            agency_id=self.agency_id,
            period_from=PbDatetimeConverter().from_datetime(self.period_from),
            reporter_type=reporter_type_converter.reversed(self.reporter_type),
        )
