import logging

from crm.agency_cabinet.common.celery.base import async_to_sync
from crm.agency_cabinet.common.server.common.structs import TaskStatuses
from crm.agency_cabinet.ord.common.consts import ReporterType
from crm.agency_cabinet.ord.server.src.celery.base import celery_app, OrdContextBoto3Task
from crm.agency_cabinet.ord.server.src.celery.tasks.transfer.export_reports import AgencyDirectReportExporter
from crm.agency_cabinet.ord.server.src.celery.tasks.transfer.import_reports import AgencyDirectReportImporter
from crm.agency_cabinet.ord.server.src.config import REPORTS_MDS_SETTINGS, MDS_SETTINGS
from crm.agency_cabinet.ord.server.src.db import db, models
from .load_client_info import load_client_info_task


__all__ = [
    'check_requested_export_reports_task',
    'check_requested_import_reports_task',
    'load_client_info_task',
    'export_agency_direct_report_task',
    'import_agency_direct_report_task',
]


LOGGER = logging.getLogger('celery.tasks')


@celery_app.task(bind=True, base=OrdContextBoto3Task)
def check_requested_export_reports_task(self: OrdContextBoto3Task):
    @async_to_sync
    async def _generate_reports():
        reports = await db.select(
            [
                models.ReportExportInfo.id,
                models.ReportExportInfo.report_id,
                models.Report.reporter_type,
                models.ReportSettings.name.label('settings_name')
            ]
        ).select_from(
            models.ReportExportInfo.join(models.Report).join(models.ReportSettings)
        ).where(
            models.ReportExportInfo.status == TaskStatuses.requested.value
        ).gino.all()

        for report in reports:
            if report['reporter_type'] == ReporterType.partner.value and report['settings_name'] == 'direct':
                export_agency_direct_report_task.apply_async(
                    (),
                    {
                        'task_id': report['id'],
                        'report_id': report['report_id']
                    },
                    queue='ord',
                )

            else:
                LOGGER.error('Not implemented')

    _generate_reports()


@celery_app.task(bind=True, base=OrdContextBoto3Task)
def check_requested_import_reports_task(self: OrdContextBoto3Task):
    @async_to_sync
    async def _import_reports():
        reports = await db.select(
            [
                models.ReportImportInfo.id,
                models.ReportImportInfo.report_id,
                models.Report.reporter_type,
                models.Report.agency_id,
                models.ReportSettings.name.label('settings_name'),
                models.S3MdsFile.name.label('mds_filename')
            ]
        ).select_from(
            models.ReportImportInfo.join(models.Report).join(models.ReportSettings).join(models.S3MdsFile)
        ).where(
            models.ReportImportInfo.status == TaskStatuses.requested.value
        ).gino.all()

        for report in reports:
            if report['reporter_type'] == ReporterType.partner.value and report['settings_name'] == 'direct':
                import_agency_direct_report_task.apply_async(
                    (),
                    {
                        'task_id': report['id'],
                        'report_id': report['report_id'],
                        'partner_id': report['agency_id'],
                        'mds_filename': report['mds_filename']
                    },
                    queue='ord',
                )

            else:
                LOGGER.error('Not implemented')

    _import_reports()


@celery_app.task(bind=True, base=OrdContextBoto3Task)
def export_agency_direct_report_task(self: OrdContextBoto3Task, task_id: str, report_id: str):
    @async_to_sync
    async def _export_report(task_id: str, report_id: str):
        await AgencyDirectReportExporter(MDS_SETTINGS, REPORTS_MDS_SETTINGS, int(task_id), int(report_id)).generate()

    _export_report(task_id, report_id)


@celery_app.task(bind=True, base=OrdContextBoto3Task)
def import_agency_direct_report_task(self: OrdContextBoto3Task, task_id: str, report_id: str, partner_id: str,
                                     mds_filename : str):
    @async_to_sync
    async def _import_report(task_id: str, report_id: str, partner_id: str, mds_filename : str):
        importer = AgencyDirectReportImporter(
            MDS_SETTINGS, REPORTS_MDS_SETTINGS, int(task_id), int(report_id), int(partner_id), mds_filename
        )
        await importer.init()
        await importer.import_report()

    _import_report(task_id, report_id, partner_id, mds_filename)
