import aioboto3
import logging
import tempfile
import typing
import xlsxwriter
import uuid
from aiobotocore.config import AioConfig

from crm.agency_cabinet.ord.server.src.db import db, models
from crm.agency_cabinet.common.server.common.config import MdsConfig
from crm.agency_cabinet.common.definitions import NamedTemporaryFileProtocol

LOGGER = logging.getLogger('celery.export_report_experiment')


class ReportExporterExperiment:
    MDS_PREFIX = 'ord_reports'

    REPORT_COLUMNS = ['Client ID',
                      'Login',
                      'Рекламная кампания',
                      'Рекомендуемая сумма (с НДС)',
                      'Идентификатор креатива',
                      'На исполнителе лежит обязанность регистрировать и репортить креатив',

                      # рекламная площадка
                      'ИНН',
                      'Тип организации',
                      'ОПФ и полное наименование',
                      'Абонентский номер мобильного телефона',
                      'Номер электронного средства платежа',
                      'Регистрационный номер либо его аналог',
                      'Номер налогоплательщика либо его аналог в стране регистрации',
                      'Код страны регистрации юрлица в соответствии с ОКСМ',
                      'Номер договора',
                      'Тип договора',
                      'Описание осуществляемых посредником-представителем действий',
                      'Сведения о предмете договора',
                      'Дата договора / доп.соглашения',
                      'Цена договора',
                      'Признак "с НДС"',
                      'Акт	Сумма (с НДС)',
                      'Признак "с НДС"',

                      # партнёр (агентство)
                      'ИНН',
                      'Тип организации',
                      'ОПФ и полное наименование',
                      'Абонентский номер мобильного телефона',
                      'Номер электронного средства платежа',
                      'Регистрационный номер либо его аналог',
                      'Номер налогоплательщика либо его аналог в стране регистрации',
                      'Код страны регистрации юрлица в соответствии с ОКСМ',
                      'Номер договора',
                      'Тип договора',
                      'Описание осуществляемых посредником-представителем действий',
                      'Сведения о предмете договора',
                      'Дата договора / доп.соглашения',
                      'Цена договора',
                      'Признак "с НДС"',
                      'Акт'
                      'Сумма (с НДС)',
                      'Признак "с НДС"',

                      # контрагент
                      'ИНН',
                      'Тип организации',
                      'ОПФ и полное наименование',
                      'Абонентский номер мобильного телефона',
                      'Номер электронного средства платежа',
                      'Регистрационный номер либо его аналог',
                      'Номер налогоплательщика либо его аналог в стране регистрации',
                      'Код страны регистрации юрлица в соответствии с ОКСМ',
                      'Номер договора',
                      'Тип договора',
                      'Описание осуществляемых посредником-представителем действий',
                      'Сведения о предмете договора',
                      'Дата договора / доп.соглашения',
                      'Цена договора',
                      'Признак "с НДС"',
                      'Акт'
                      'Сумма (с НДС)',
                      'Признак "с НДС"'
                      ]

    def __init__(self, mds_cfg: MdsConfig, report_id: int):
        self._report_id = report_id
        self._bucket = 'agency-cabinet-common'
        self._mds_cfg = mds_cfg

    async def generate(self):
        try:
            await self._update_status('stub')
            report_file, report_name = await self._make_report()
            with report_file:
                mds_filename = await self._upload_to_mds(report_file, report_name)
                await self._update_status_and_save_mds_filename(mds_filename, report_name)
        except Exception as ex:
            LOGGER.exception('Error during report generation: %s', ex)
            await self._update_status('error')
            raise ex

    async def _update_status(self, status: str):
        pass

    async def _make_report(self) -> typing.Tuple[NamedTemporaryFileProtocol, str]:
        report_file = tempfile.NamedTemporaryFile()
        with xlsxwriter.Workbook(report_file.name) as workbook:
            worksheet_total = workbook.add_worksheet('Отчёт')

            worksheet_total.write_row(
                'A1',
                self.REPORT_COLUMNS
            )

            row_index=2
            async for row in self._fetch_data_from_db():
                worksheet_total.write_row(f'A{row_index}', [
                    row.client_id,
                    row.client_login,
                    row.campaign_name,
                    row.suggested_amount,
                    row.campaign_eid,
                    True,

                    # рп
                    'inn',
                    'type',
                    'name',
                    'mobile phone',
                    '',
                    'reg number',
                    '',
                    '',
                    'eid',
                    '',
                    '',
                    '',
                    '',
                    '',
                    '',
                    row.ad_distributor_act_eid,
                    row.ad_distributor_act_amount,
                    '',

                    # партнёр
                    'inn',
                    'type',
                    'name',
                    'mobile phone',
                    '',
                    'reg number',
                    '',
                    '',
                    'eid',
                    '',
                    '',
                    '',
                    '',
                    '',
                    '',
                    row.partner_act_eid,
                    row.partner_act_amount,
                    ''
                ])
                row_index += 1

        report_file.seek(0)
        return report_file, report_file.name

    async def _fetch_data_from_db(self) -> typing.List[models.ClientRow]:
        ContractorOrganization = models.Organization.alias()
        PartnerContract = models.Contract.alias()
        PartnerAct = models.Act.alias()
        AdDistributorAct = models.Act.alias()

        query = db.select([
            models.Client.client_id.label('client_id'),
            models.Client.login.label('client_login'),
            models.Campaign.name.label('campaign_name'),
            models.Campaign.campaign_eid.label('campaign_eid'),
            db.func.coalesce(models.ClientRow.suggested_amount, 0).label('suggested_amount'),

            ContractorOrganization.inn,
            ContractorOrganization.type,
            ContractorOrganization.is_ors,
            ContractorOrganization.is_rr,
            ContractorOrganization.name,
            ContractorOrganization.mobile_phone,
            ContractorOrganization.epay_number,
            ContractorOrganization.reg_number,
            ContractorOrganization.alter_inn,
            ContractorOrganization.oksm_number,

            PartnerContract.contract_eid,
            PartnerContract.type,
            PartnerContract.action_type,
            PartnerContract.subject_type,
            PartnerContract.date,
            PartnerContract.amount,
            PartnerContract.is_vat,

            PartnerAct.act_eid.label('partner_act_eid'),
            PartnerAct.amount.label('partner_act_amount'),
            PartnerAct.is_vat,

            AdDistributorAct.act_eid.label('ad_distributor_act_eid'),
            AdDistributorAct.amount.label('ad_distributor_act_amount'),
            AdDistributorAct.is_vat
        ]).select_from(
            models.ClientRow
                .join(models.Client, models.Client.id == models.ClientRow.client_id)
                .outerjoin(AdDistributorAct, models.ClientRow.ad_distributor_act_id == AdDistributorAct.id)
                .outerjoin(PartnerAct, models.ClientRow.partner_act_id == PartnerAct.id)
                .outerjoin(models.Campaign, models.ClientRow.campaign_id == models.Campaign.id)
                .outerjoin(PartnerContract, models.ClientRow.partner_contract_id == PartnerContract.id)
                .outerjoin(ContractorOrganization, PartnerContract.contractor_id == ContractorOrganization.id)
        ).where(
            models.Client.report_id == self._report_id
        ).order_by(
            models.ClientRow.id
        )

        async with db.transaction():
            async for row in query.gino.iterate():
                yield row

    async def _upload_to_mds(self, report_file: NamedTemporaryFileProtocol, report_filename: str) -> str:
        mds_filename = f'{self.MDS_PREFIX}/{str(uuid.uuid4())}-{report_filename}.xls'

        boto3_session = aioboto3.Session(
            aws_access_key_id=self._mds_cfg['access_key_id'],
            aws_secret_access_key=self._mds_cfg['secret_access_key']
        )

        # TODO: don't create session each time
        # TODO: add aioboto3.session to BotoSessionBoundedMixin and pass it through task?

        async with boto3_session.resource('s3', endpoint_url=self._mds_cfg['endpoint_url'], config=AioConfig(s3={'addressing_style': 'virtual'})) as s3:
            obj = await s3.Object(self._bucket, mds_filename)
            await obj.put(
                Body=report_file.file,
                ContentType='application/vnd.ms-excel; charset=utf-8',
                ContentDisposition=f'attachment; filename={report_filename}.xls'
            )

        return mds_filename

    async def _update_status_and_save_mds_filename(self, mds_filename: str, display_name: str):
        pass
