import logging
from datetime import datetime, timezone
from yt.wrapper import YtClient
from yql.api.v1.client import YqlClient

from crm.agency_cabinet.common.celery.base import async_to_sync
from crm.agency_cabinet.ord.server.src import config
from crm.agency_cabinet.ord.server.src.celery.base import celery_app as celery
from crm.agency_cabinet.ord.server.src.config import OrdConfig
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery

from .loader import ReportsLoader


LOGGER = logging.getLogger('celery.load_reports_task')

NETTING_ACT_SOURCE_ROOT_PATH = '//home/balance/prod/acts/tariffied_acts_no_netting_agg'
DIRECT_SOURCE_ROOT_PATH = '//home/direct/mysql-sync/current'


@celery.task(bind=True)
def load_reports_task(self, period_from: str = None, agency_id: str = None, force_load: bool = False):
    yt_client = YtClient(proxy='hahn', token=config.YT_CONFIG['TOKEN'])
    yql_client = YqlClient(token=config.YQL_CONFIG['TOKEN'])

    if period_from is not None:
        period_from = datetime.strptime(period_from, '%Y-%m-%d').replace(tzinfo=timezone.utc)
        netting_act_table_path = f'{NETTING_ACT_SOURCE_ROOT_PATH}/{period_from.year}-{datetime.strftime(period_from, "%m")}'
    else:
        all_tables = yt_client.search(
            NETTING_ACT_SOURCE_ROOT_PATH,
            node_type=['table'],
        )
        netting_act_table_path = sorted(all_tables)[-1]
        table_path_stripped = netting_act_table_path.rstrip('/')
        date = table_path_stripped[table_path_stripped.rfind('/') + 1:]
        year, month = int(date[:4]), int(date[5:])
        period_from = datetime(year=year, month=month, day=1, tzinfo=timezone.utc)

    @async_to_sync
    async def load():
        cfg = OrdConfig.from_environ()
        sd = ServiceDiscovery(cfg.amqp_url)

        async with sd:
            await ReportsLoader(
                yt_client, yql_client, netting_act_table_path, DIRECT_SOURCE_ROOT_PATH, period_from, agency_id, sd).load(force_load)

    load()
