import logging
from collections import defaultdict

from crm.agency_cabinet.common.consts import Currencies, VAT_FULL
from crm.agency_cabinet.ord.common.consts import ReportStatuses, ReporterType
from crm.agency_cabinet.common.yt.synchronizers import BaseSynchronizer
from crm.agency_cabinet.ord.server.src.db import models
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery
from crm.agency_cabinet.grants.common.consts.partner import PartnerType

LOGGER = logging.getLogger('celery.load_reports.synchronizer')


class ReportsSynchronizer(BaseSynchronizer):

    def __init__(self, period_from, sd: ServiceDiscovery):
        self.period_from = period_from
        self.sd = sd

    async def process_data(self, rows: list[tuple], *args, **kwargs) -> bool:
        list_partners_response = await self.sd.grants.list_partners(type=PartnerType.agency)
        agency_id_to_partner_id = {
            e.external_id: e.partner_id for e in list_partners_response.partners
        }

        reports = {}
        clients = defaultdict(lambda: defaultdict(dict))
        clients_data = defaultdict(lambda: defaultdict(list))
        acts = defaultdict(lambda: defaultdict(int))
        campaigns = defaultdict(lambda: defaultdict(dict))

        for row in rows:
            agency_id, client_id, amount, act_id, act_eid, login, name, campaign_eid, campaign_name = row
            agency_inn, contract_eid, vat = None, 'stub', VAT_FULL

            if agency_id not in reports:
                reports[agency_id] = {
                    'contract_eid': contract_eid,
                    'agency_inn': agency_inn,
                    'vat': vat if vat == 0 else VAT_FULL,
                    'currency': Currencies.rub.value
                }

            clients[agency_id][client_id] = {
                'login': login,
                'name': name
            }

            clients_data[agency_id][client_id].append((
                act_eid,
                amount,
                campaign_eid
            ))

            acts[agency_id][act_eid] += amount

            campaigns[agency_id][campaign_eid] = {
                'campaign_name': campaign_name,
                'client_id': client_id
            }

        report_settings = await models.ReportSettings.query.where(
            models.ReportSettings.name == 'direct'
        ).gino.first()

        for agency_id, data in reports.items():
            partner_id = agency_id_to_partner_id.get(str(agency_id))
            if partner_id is None:
                LOGGER.warning('Unknown agency_id: %s', agency_id)
                continue
            report = await models.Report.create(
                agency_id=partner_id,
                settings_id=report_settings.id,
                period_from=self.period_from,
                status=ReportStatuses.draft.value,
                reporter_type=ReporterType.partner.value,
                external_id=str(agency_id)
            )

            rows = []
            for act_eid, act_amount in acts[agency_id].items():
                rows.append({
                    'report_id': report.id,
                    'act_eid': act_eid,
                    'amount': act_amount,
                    'is_valid': True,
                })

            acts_db = await models.Act.bulk_insert(rows)
            act_ids = {act.act_eid: act.id for act in acts_db}

            rows = []
            for client_id, client_data in clients[agency_id].items():
                rows.append({
                    'client_id': str(client_id),
                    'report_id': report.id,
                    'login': client_data['login'],
                    'name': client_data['name']
                })

            clients_db = await models.Client.bulk_insert(rows)
            client_ids = {client.client_id: client.id for client in clients_db}

            rows = []
            for campaign_eid, campaign_data in campaigns[agency_id].items():
                rows.append({
                    'report_id': report.id,
                    'client_id': client_ids[str(campaign_data['client_id'])],
                    'campaign_eid': str(campaign_eid),
                    'name': str(campaign_data['campaign_name'])
                })

            campaigns_db = await models.Campaign.bulk_insert(rows)
            campaign_ids = {campaign.campaign_eid: campaign.id for campaign in campaigns_db}

            client_rows = []
            for client_id, items in clients_data[agency_id].items():
                for act_eid, suggested_amount, campaign_eid in items:
                    client_rows.append({
                        'client_id': client_ids[str(client_id)],
                        'suggested_amount': suggested_amount,
                        'ad_distributor_act_id': act_ids[act_eid],
                        'campaign_id': campaign_ids[str(campaign_eid)]
                    })

            await models.ClientRow.bulk_insert(client_rows)
        return True
