import typing

from datetime import datetime
from gino.loader import ColumnLoader
from sqlalchemy import and_, distinct, desc, asc, text, or_

from crm.agency_cabinet.ord.server.src.db import models, db
from crm.agency_cabinet.ord.common.structs import ReportStatuses, ReportSort, ReportColumns, SortTypes


ENUM_COLUMN_ORM_COLUMN = {
    ReportColumns.status: models.Report.status,
    ReportColumns.clients_count: text('clients_count'),
    ReportColumns.sending_date: models.Report.sending_date
}


def _sort_struct_to_query(sort: typing.List[ReportSort]):
    res = []
    for s in sort:
        col = ENUM_COLUMN_ORM_COLUMN.get(s.column)
        if col is not None:
            res.append(asc(col) if s.type == SortTypes.asc else desc(col))
    return res


def build_get_reports(
    agency_id: int = None,
    search_query: str = None,  # noqa TODO: ???
    period_from: datetime = None,
    period_to: datetime = None,
    status: ReportStatuses = None,
    sort: list[ReportSort] = (),
    report_id: int = None,
    limit: int = None,
    offset: int = None,
):
    clients_count = db.func.count(distinct(models.Client.client_id)).label('clients_count')
    campaigns_count = db.func.count(distinct(models.ClientRow.campaign_id)).label('campaigns_count')

    q = db.select(
        [
            models.Report,
            models.ReportSettings,
            clients_count,
            campaigns_count
        ]
    ).select_from(
        models.Report.outerjoin(models.Client).outerjoin(models.ClientRow).join(models.ReportSettings)
    )

    conditions = [models.Report.is_deleted.is_(False)]

    if agency_id is not None:
        conditions.append(models.Report.agency_id == agency_id)

    if period_from is not None:
        conditions.append(models.Report.period_from >= period_from)

    if period_to is not None:
        conditions.append(models.Report.period_from < period_to)

    if status is not None and status != ReportStatuses.all:
        conditions.append(models.Report.status == status.value)

    if report_id is not None:
        conditions.append(models.Report.id == report_id)

    if conditions:
        q = q.where(and_(*conditions))

    if limit is not None:
        q = q.limit(limit)

    if offset is not None:
        q = q.offset(offset)

    q = q.group_by(models.Report.id, models.ReportSettings.id).order_by(*_sort_struct_to_query(sort))

    return q.gino.load((models.Report, models.ReportSettings, ColumnLoader(clients_count), ColumnLoader(campaigns_count)))


# noinspection PyPep8Naming
def build_get_client_rows_query(
    report_id: int = None,
    client_id: int = None,
    search_query: str = None,
    only_size: bool = False,
    only_max_update: bool = False
):
    AdDistributorOrganization = models.Organization.alias()  # Площадка
    AdDistributorPartnerOrganization = models.Organization.alias()  # Партнёр
    PartnerClientOrganization = models.Organization.alias()  # Контрагент партнёра
    AdvertiserContractorOrganization = models.Organization.alias()  # Контрагент конечного рекламодателя
    AdvertiserOrganization = models.Organization.alias()  # Конечный рекламодатель

    AdDistributorContract = models.Contract.alias()  # contractor_id - площадка, client_id - партнёр
    PartnerContract = models.Contract.alias()  # contractor_id - партнёр, client_id - контрагент партнёра
    AdvertiserContract = models.Contract.alias()  # contractor_id - конечный рекламодатель, client_id - контрагент конечного рекламодателя

    PartnerAct = models.Act.alias()  # акт, высталвяется партнёром
    AdDistributorAct = models.Act.alias()  # акт, выставляется площадкой

    if only_size:
        query = db.select([db.func.count()])

    elif only_max_update:
        query = db.select([
            db.func.greatest(
                db.func.max(models.Client.updated_at),
                db.func.max(models.ClientRow.updated_at),
                db.func.max(models.Campaign.updated_at),
                db.func.max(PartnerAct.updated_at),
                db.func.max(AdDistributorAct.updated_at),
                db.func.max(AdDistributorContract.updated_at),
                db.func.max(PartnerContract.updated_at),
                db.func.max(AdvertiserContract.updated_at),
                db.func.max(AdDistributorOrganization.updated_at),
                db.func.max(AdDistributorPartnerOrganization.updated_at),
                db.func.max(PartnerClientOrganization.updated_at),
                db.func.max(AdvertiserContractorOrganization.updated_at),
                db.func.max(AdvertiserOrganization.updated_at),
            ).label('updated_at')
        ])
    else:
        query = db.select([
            models.ClientRow.id,
            models.ClientRow.suggested_amount,

            # клиент
            models.Client.id.label('client_id'),
            models.Client.client_id.label('client_eid'),
            models.Client.login.label('client_login'),
            models.Client.name.label('client_name'),

            # кампания
            models.Campaign.id.label('campaign_id'),
            models.Campaign.name.label('campaign_name'),
            models.Campaign.campaign_eid.label('campaign_eid'),

            # Площадка
            AdDistributorOrganization.id.label('ad_distributor_org_id'),
            AdDistributorOrganization.inn.label('ad_distributor_org_inn'),
            AdDistributorOrganization.type.label('ad_distributor_org_type'),
            AdDistributorOrganization.name.label('ad_distributor_org_name'),
            AdDistributorOrganization.is_rr.label('ad_distributor_org_is_rr'),
            AdDistributorOrganization.is_ors.label('ad_distributor_org_is_ors'),
            AdDistributorOrganization.mobile_phone.label('ad_distributor_org_mobile_phone'),
            AdDistributorOrganization.epay_number.label('ad_distributor_org_epay_number'),
            AdDistributorOrganization.reg_number.label('ad_distributor_org_reg_number'),
            AdDistributorOrganization.alter_inn.label('ad_distributor_org_alter_inn'),
            AdDistributorOrganization.oksm_number.label('ad_distributor_org_oksm_number'),
            AdDistributorOrganization.rs_url.label('ad_distributor_org_rs_url'),

            # Договор между площадкой и партнёром
            AdDistributorContract.id.label('ad_distributor_contract_id'),
            AdDistributorContract.contract_eid.label('ad_distributor_contract_eid'),
            AdDistributorContract.is_reg_report.label('ad_distributor_contract_is_reg_report'),
            AdDistributorContract.type.label('ad_distributor_contract_type'),
            AdDistributorContract.action_type.label('ad_distributor_contract_action_type'),
            AdDistributorContract.subject_type.label('ad_distributor_contract_subject_type'),
            AdDistributorContract.date.label('ad_distributor_contract_date'),
            AdDistributorContract.amount.label('ad_distributor_contract_amount'),
            AdDistributorContract.is_vat.label('ad_distributor_contract_is_vat'),

            # Акт площадки
            AdDistributorAct.id.label('ad_distributor_act_id'),
            AdDistributorAct.act_eid.label('ad_distributor_act_eid'),
            AdDistributorAct.amount.label('ad_distributor_act_amount'),
            AdDistributorAct.is_vat.label('ad_distributor_act_is_vat'),

            # Партнёр
            AdDistributorPartnerOrganization.id.label('ad_distributor_partner_org_id'),
            AdDistributorPartnerOrganization.inn.label('ad_distributor_partner_org_inn'),
            AdDistributorPartnerOrganization.type.label('ad_distributor_partner_org_type'),
            AdDistributorPartnerOrganization.name.label('ad_distributor_partner_org_name'),
            AdDistributorPartnerOrganization.is_rr.label('ad_distributor_partner_org_is_rr'),
            AdDistributorPartnerOrganization.is_ors.label('ad_distributor_partner_org_is_ors'),
            AdDistributorPartnerOrganization.mobile_phone.label('ad_distributor_partner_org_mobile_phone'),
            AdDistributorPartnerOrganization.epay_number.label('ad_distributor_partner_org_epay_number'),
            AdDistributorPartnerOrganization.reg_number.label('ad_distributor_partner_org_reg_number'),
            AdDistributorPartnerOrganization.alter_inn.label('ad_distributor_partner_org_alter_inn'),
            AdDistributorPartnerOrganization.oksm_number.label('ad_distributor_partner_org_oksm_number'),
            AdDistributorPartnerOrganization.rs_url.label('ad_distributor_partner_org_rs_url'),

            # Акт партнёра
            PartnerAct.id.label('partner_act_id'),
            PartnerAct.act_eid.label('partner_act_eid'),
            PartnerAct.amount.label('partner_act_amount'),
            PartnerAct.is_vat.label('partner_act_is_vat'),

            # Договор между партнёром и его контрагентом
            models.ClientRow.partner_contract_id,
            PartnerContract.contract_eid.label('partner_contract_eid'),
            PartnerContract.is_reg_report.label('partner_contract_is_reg_report'),
            PartnerContract.type.label('partner_contract_type'),
            PartnerContract.action_type.label('partner_contract_action_type'),
            PartnerContract.subject_type.label('partner_contract_subject_type'),
            PartnerContract.date.label('partner_contract_date'),
            PartnerContract.amount.label('partner_contract_amount'),
            PartnerContract.is_vat.label('partner_contract_is_vat'),

            # Контрагент партнёра
            PartnerClientOrganization.id.label('partner_client_org_id'),
            PartnerClientOrganization.inn.label('partner_client_org_inn'),
            PartnerClientOrganization.type.label('partner_client_org_type'),
            PartnerClientOrganization.name.label('partner_client_org_name'),
            PartnerClientOrganization.is_rr.label('partner_client_org_is_rr'),
            PartnerClientOrganization.is_ors.label('partner_client_org_is_ors'),
            PartnerClientOrganization.mobile_phone.label('partner_client_org_mobile_phone'),
            PartnerClientOrganization.epay_number.label('partner_client_org_epay_number'),
            PartnerClientOrganization.reg_number.label('partner_client_org_reg_number'),
            PartnerClientOrganization.alter_inn.label('partner_client_org_alter_inn'),
            PartnerClientOrganization.oksm_number.label('partner_client_org_oksm_number'),
            PartnerClientOrganization.rs_url.label('partner_client_org_rs_url'),

            # Договор между конечным рекламодателем и его исполнителем
            models.ClientRow.advertiser_contract_id,
            AdvertiserContract.contract_eid.label('advertiser_contract_eid'),
            AdvertiserContract.is_reg_report.label('advertiser_contract_is_reg_report'),
            AdvertiserContract.type.label('advertiser_contract_type'),
            AdvertiserContract.action_type.label('advertiser_contract_action_type'),
            AdvertiserContract.subject_type.label('advertiser_contract_subject_type'),
            AdvertiserContract.date.label('advertiser_contract_date'),
            AdvertiserContract.amount.label('advertiser_contract_amount'),
            AdvertiserContract.is_vat.label('advertiser_contract_is_vat'),

            # Контрагент конечного рекламодателя
            AdvertiserContractorOrganization.id.label('advertiser_contractor_org_id'),
            AdvertiserContractorOrganization.inn.label('advertiser_contractor_org_inn'),
            AdvertiserContractorOrganization.type.label('advertiser_contractor_org_type'),
            AdvertiserContractorOrganization.name.label('advertiser_contractor_org_name'),
            AdvertiserContractorOrganization.is_rr.label('advertiser_contractor_org_is_rr'),
            AdvertiserContractorOrganization.is_ors.label('advertiser_contractor_org_is_ors'),
            AdvertiserContractorOrganization.mobile_phone.label('advertiser_contractor_org_mobile_phone'),
            AdvertiserContractorOrganization.epay_number.label('advertiser_contractor_org_epay_number'),
            AdvertiserContractorOrganization.reg_number.label('advertiser_contractor_org_reg_number'),
            AdvertiserContractorOrganization.alter_inn.label('advertiser_contractor_org_alter_inn'),
            AdvertiserContractorOrganization.oksm_number.label('advertiser_contractor_org_oksm_number'),
            AdvertiserContractorOrganization.rs_url.label('advertiser_contractor_org_rs_url'),

            # Конечный рекламодатель
            AdvertiserOrganization.id.label('advertiser_org_id'),
            AdvertiserOrganization.inn.label('advertiser_org_inn'),
            AdvertiserOrganization.type.label('advertiser_org_type'),
            AdvertiserOrganization.name.label('advertiser_org_name'),
            AdvertiserOrganization.is_rr.label('advertiser_org_is_rr'),
            AdvertiserOrganization.is_ors.label('advertiser_org_is_ors'),
            AdvertiserOrganization.mobile_phone.label('advertiser_org_mobile_phone'),
            AdvertiserOrganization.epay_number.label('advertiser_org_epay_number'),
            AdvertiserOrganization.reg_number.label('advertiser_org_reg_number'),
            AdvertiserOrganization.alter_inn.label('advertiser_org_alter_inn'),
            AdvertiserOrganization.oksm_number.label('advertiser_org_oksm_number'),
            AdvertiserOrganization.rs_url.label('advertiser_org_rs_url'),
        ])

    query = query.select_from(
        models.ClientRow
        .join(models.Client, models.Client.id == models.ClientRow.client_id)
        .outerjoin(models.Campaign, models.ClientRow.campaign_id == models.Campaign.id)

        .outerjoin(AdDistributorContract, models.ClientRow.ad_distributor_contract_id == AdDistributorContract.id)
        .outerjoin(AdDistributorOrganization, AdDistributorContract.contractor_id == AdDistributorOrganization.id)
        .outerjoin(AdDistributorPartnerOrganization,
                   AdDistributorContract.client_id == AdDistributorPartnerOrganization.id)
        .outerjoin(AdDistributorAct, models.ClientRow.ad_distributor_act_id == AdDistributorAct.id)

        .outerjoin(PartnerContract, models.ClientRow.partner_contract_id == PartnerContract.id)
        .outerjoin(PartnerClientOrganization, PartnerContract.client_id == PartnerClientOrganization.id)
        .outerjoin(PartnerAct, models.ClientRow.partner_act_id == PartnerAct.id)

        .outerjoin(AdvertiserContract, models.ClientRow.advertiser_contract_id == AdvertiserContract.id)
        .outerjoin(AdvertiserOrganization, AdvertiserContract.client_id == AdvertiserOrganization.id)
        .outerjoin(AdvertiserContractorOrganization,
                   AdvertiserContract.contractor_id == AdvertiserContractorOrganization.id)
    )

    if report_id:
        query = query.where(models.Client.report_id == report_id)

    if client_id:
        query = query.where(models.ClientRow.client_id == client_id)

    if search_query:
        search_query = '%{}%'.format(search_query.lower())

        query = query.where(
            or_(
                models.Campaign.campaign_eid.like(search_query),
                AdDistributorOrganization.inn.like(search_query),
                AdDistributorPartnerOrganization.inn.like(search_query),
                PartnerClientOrganization.inn.like(search_query),
                AdvertiserContractorOrganization.inn.like(search_query),
                AdvertiserOrganization.inn.like(search_query),
            )
        )

    return query
