import asyncio
from marshmallow import ValidationError

from dataclasses import dataclass
from crm.agency_cabinet.ord.server.src.db import models, db
from crm.agency_cabinet.ord.common import structs, schemas
from crm.agency_cabinet.ord.common.exceptions import NoSuchActException, UnsuitableReportException


@dataclass
class AddAct:
    async def __call__(self, request: structs.AddActInput) -> structs.Act:
        report = await models.Report.query.where(models.Report.id == request.report_id).gino.first()

        if report is None or report.agency_id != request.agency_id:
            raise UnsuitableReportException()

        try:
            schemas.Act().dump(request)
            is_valid = True
        except ValidationError:
            is_valid = False

        act = await models.Act.create(
            report_id=request.report_id,
            act_eid=request.act_eid,
            amount=request.amount,
            is_vat=request.is_vat,
            is_valid=is_valid
        )

        return structs.Act(
            act_id=act.id,
            act_eid=act.act_eid,
            amount=act.amount,
            is_vat=act.is_vat,
        )


@dataclass
class EditAct:
    async def __call__(self, request: structs.EditActInput):
        act = await models.Act.query.where(models.Act.id == request.act_id).gino.first()

        if act is None:
            raise NoSuchActException()

        if act.report_id != request.report_id:
            raise UnsuitableReportException()

        updated_data = {}

        if request.act_eid is not None:
            updated_data['act_eid'] = request.act_eid

        if request.amount is not None:
            updated_data['amount'] = request.amount

        if request.amount is not None:
            updated_data['is_vat'] = request.is_vat

        try:
            schemas.Act().dump(request)
            updated_data['is_valid'] = True
        except ValidationError:
            updated_data['is_valid'] = False

        await models.Act.update.values(**updated_data).where(
            models.Act.id == request.act_id
        ).gino.status(read_only=False, reuse=False)


@dataclass
class GetActs:
    async def __call__(self, request: structs.GetActsInput) -> structs.ActList:

        report = await models.Report.query.where(models.Report.id == request.report_id).gino.first()

        if report is None or report.agency_id != request.agency_id:
            raise UnsuitableReportException()

        query = models.Act.query.where(
            models.Act.report_id == request.report_id
        )

        size_query = db.select(
            [db.func.count()]
        ).select_from(models.Act).where(
            models.Act.report_id == request.report_id
        )

        if request.search_query:
            search_query = '%{}%'.format(request.search_query.lower())

            query = query.where(
                db.func.lower(models.Act.act_eid).like(search_query),
            )

            size_query = size_query.where(
                db.func.lower(models.Act.act_eid).like(search_query),
            )

        if request.limit:
            query = query.limit(request.limit)

        if request.offset:
            query = query.offset(request.offset)

        acts = await query.order_by(models.Act.id.desc()).gino.all()
        get_size_task = asyncio.create_task(size_query.gino.scalar())

        return structs.ActList(
            size=await get_size_task,
            acts=[structs.Act(
                act_id=act.id,
                act_eid=act.act_eid,
                amount=act.amount,
                is_vat=act.is_vat
            ) for act in acts]
        )
