from dataclasses import dataclass
from sqlalchemy import or_, and_
from crm.agency_cabinet.ord.server.src.db import db, models
from crm.agency_cabinet.ord.common import structs


@dataclass
class GetCampaigns:
    async def __call__(self, request: structs.GetCampaignsInput) -> structs.CampaignList:
        query = db.select(
            [
                models.Campaign.id,
                models.Campaign.campaign_eid,
                models.Campaign.name,
            ]
        ).select_from(models.Campaign.join(models.Report)).where(
            and_(
                models.Campaign.client_id == request.client_id,
                models.Campaign.report_id == request.report_id,
                models.Report.agency_id == request.agency_id,
            )
        )

        if request.search_query:
            search_query = '%{}%'.format(request.search_query.lower())

            query = query.where(
                or_(
                    db.func.lower(models.Campaign.name).ilike(search_query),
                    db.func.lower(models.Campaign.campaign_eid).like(search_query),
                )
            )

        if request.limit:
            query = query.limit(request.limit)

        if request.offset:
            query = query.offset(request.offset)

        campaigns = await query.order_by(models.Campaign.id.desc()).gino.all()

        return structs.CampaignList(
            size=len(campaigns),
            campaigns=[structs.Campaign(
                id=campaign.id,
                name=campaign.name,
                campaign_eid=campaign.campaign_eid
            ) for campaign in campaigns]
        )
