import asyncio
import asyncpg.exceptions
import typing

from dataclasses import dataclass
from crm.agency_cabinet.ord.server.src.db import db, models
from crm.agency_cabinet.ord.server.src.db.queries import build_get_client_rows_query
from crm.agency_cabinet.ord.common import structs
from crm.agency_cabinet.ord.common.exceptions import UniqueViolationClientRowException, NoSuchClientRowException, \
    UnsuitableReportException, UnsuitableAgencyException, UnsuitableClientException


@dataclass
class GetClientRows:

    async def __call__(self, request: structs.GetClientRowsInput) -> structs.ClientRowsList:
        report = await models.Report.query.where(
            models.Report.id == request.report_id
        ).gino.first()

        if report is None:
            raise UnsuitableReportException()
        if report.agency_id != request.agency_id:
            raise UnsuitableAgencyException()

        # todo: need to extend query for ad_distributor reports and take partner_organization from client_row.ad_distributor_contract.client_organization
        query = build_get_client_rows_query(request.report_id, request.client_id, request.search_query)
        size_query = build_get_client_rows_query(
            request.report_id, request.client_id, request.search_query, only_size=True
        )

        get_size_task = asyncio.create_task(size_query.gino.scalar())

        if request.limit:
            query = query.limit(request.limit)

        if request.offset:
            query = query.offset(request.offset)

        query = query.order_by(models.ClientRow.updated_at.desc())
        rows = await query.gino.all()

        return structs.ClientRowsList(
            size=await get_size_task,
            rows=[
                structs.ClientRow(
                    id=row.id,
                    suggested_amount=row.suggested_amount,
                    campaign=structs.Campaign(
                        id=row.campaign_id,
                        campaign_eid=row.campaign_eid,
                        name=row.campaign_name,
                    ),
                    ad_distributor_organization=self._get_organization(row, 'ad_distributor_org'),
                    ad_distributor_partner_organization=self._get_organization(row, 'ad_distributor_partner_org'),
                    partner_client_organization=self._get_organization(row, 'partner_client_org'),
                    advertiser_contractor_organization=self._get_organization(row, 'advertiser_contractor_org'),
                    advertiser_organization=self._get_organization(row, 'advertiser_org'),
                    ad_distributor_contract=self._get_contract(row, 'ad_distributor_contract'),
                    ad_distributor_partner_contract=self._get_contract(row, 'partner_contract'),
                    advertiser_contract=self._get_contract(row, 'advertiser_contract'),
                    ad_distributor_act=self._get_act(row, 'ad_distributor_act'),
                    ad_distributor_partner_act=self._get_act(row, 'partner_act'),
                ) for row in rows
            ]
        )

    @staticmethod
    def _get_organization(row: dict, prefix: str) -> typing.Optional[structs.Organization]:
        if getattr(row, f'{prefix}_id') is None:
            return None

        return structs.Organization(
            id=getattr(row, f'{prefix}_id'),
            type=getattr(row, f'{prefix}_type'),
            name=getattr(row, f'{prefix}_name'),
            inn=getattr(row, f'{prefix}_inn'),
            is_rr=getattr(row, f'{prefix}_is_rr'),
            is_ors=getattr(row, f'{prefix}_is_ors'),
            mobile_phone=getattr(row, f'{prefix}_mobile_phone'),
            epay_number=getattr(row, f'{prefix}_epay_number'),
            reg_number=getattr(row, f'{prefix}_reg_number'),
            alter_inn=getattr(row, f'{prefix}_alter_inn'),
            oksm_number=getattr(row, f'{prefix}_oksm_number'),
            rs_url=getattr(row, f'{prefix}_rs_url'),
        )

    @staticmethod
    def _get_contract(row: dict, prefix: str) -> typing.Optional[structs.Contract]:
        if getattr(row, f'{prefix}_id') is None:
            return None

        return structs.Contract(
            id=getattr(row, f'{prefix}_id'),
            contract_eid=getattr(row, f'{prefix}_eid'),
            is_reg_report=getattr(row, f'{prefix}_is_reg_report'),
            type=getattr(row, f'{prefix}_type'),
            action_type=getattr(row, f'{prefix}_action_type'),
            subject_type=getattr(row, f'{prefix}_subject_type'),
            date=getattr(row, f'{prefix}_date'),
            amount=getattr(row, f'{prefix}_amount'),
            is_vat=getattr(row, f'{prefix}_is_vat'),
        )

    @staticmethod
    def _get_act(row: dict, prefix: str) -> typing.Optional[structs.Act]:
        if getattr(row, f'{prefix}_id') is None:
            return None

        return structs.Act(
            act_id=getattr(row, f'{prefix}_id'),
            act_eid=getattr(row, f'{prefix}_eid'),
            amount=getattr(row, f'{prefix}_amount'),
            is_vat=getattr(row, f'{prefix}_is_vat'),
        )


@dataclass
class EditClientRow:
    async def __call__(self, request: structs.EditClientRowInput):
        client_row = await db.select([
            models.Report.agency_id,
            models.Client.report_id,
            models.ClientRow.client_id,
        ]).select_from(
            models.ClientRow.join(models.Client).join(models.Report)
        ).where(
            models.ClientRow.id == request.row_id
        ).gino.first()

        if not client_row:
            raise NoSuchClientRowException()

        if client_row.report_id != request.report_id:
            raise UnsuitableReportException()

        if client_row.client_id != request.client_id:
            raise UnsuitableClientException()

        if client_row.agency_id != request.agency_id:
            raise UnsuitableAgencyException()

        # todo: need to use ReportSetings
        # if client_row.ad_distributor_id == AD_DISTRIBUTOR_ID_YANDEX and (request.ad_distributor_act_id is not None or request.campaign_eid is not None):
        #     raise UnsuitableReportException()

        update_data = {}

        # if request.ad_distributor_act_id is not None:
        #     ad_distributor_act = await models.AdDistributorAct.query.where(
        #         models.AdDistributorAct.id == request.ad_distributor_act_id
        #     ).gino.first()
        #
        #     if ad_distributor_act is None or ad_distributor_act.report_id != request.report_id:
        #         raise OrdException()
        #
        #     update_data['ad_distributor_act_id'] = request.ad_distributor_act_id

        # if request.client_act_id is not None:
        #     client_act = await models.ClientAct.query.where(
        #         models.ClientAct.id == request.client_act_id
        #     ).gino.first()
        #
        #     if client_act is None or client_act.report_id != request.report_id:
        #         raise OrdException()
        #
        #     update_data['client_act_id'] = request.client_act_id

        # if request.client_contract_id is not None:
        #     client_contract = await models.ClientContract.query.where(
        #         models.ClientContract.id == request.client_contract_id
        #     ).gino.first()
        #
        #     if client_contract is None:
        #         raise OrdException()
        #
        #     update_data['client_contract_id'] = request.client_contract_id

        try:
            async with db.transaction(read_only=False, reuse=False):
                campaign = None
                if request.campaign_eid is not None:
                    campaign = await models.Campaign.update_or_create(
                        filter_columns={
                            'campaign_eid': request.campaign_eid,
                            'report_id': request.report_id,
                            'client_id': request.client_id,
                        },
                        update_columns=dict(
                            campaign_eid=request.campaign_eid,
                            name=request.campaign_name,
                            client_id=request.client_id,
                            report_id=request.report_id,
                        )
                    )

                if campaign is not None:
                    update_data['campaign_id'] = campaign.id

                await models.ClientRow.update.values(**update_data).where(
                    models.ClientRow.id == request.row_id).gino.status(read_only=False, reuse=False)

        except asyncpg.exceptions.UniqueViolationError:
            raise UniqueViolationClientRowException()
