import asyncpg.exceptions

from dataclasses import dataclass
from sqlalchemy import and_, or_
from crm.agency_cabinet.ord.server.src.db import db, models
from crm.agency_cabinet.ord.common import structs
from crm.agency_cabinet.ord.common.exceptions import UnsuitableReportException, UnsuitableAgencyException, \
    UnsuitableClientException, NoSuchReportException, UniqueViolationClientException, ForbiddenByReportSettingsException


@dataclass
class GetReportClientsInfo:
    async def __call__(self, request: structs.GetReportClientsInfoInput) -> structs.ClientsInfoList:
        AdDistributorOrganization = models.Organization.alias()
        AdDistributorPartnerOrganization = models.Organization.alias()
        PartnerClientOrganization = models.Organization.alias()
        AdvertiserContractorOrganization = models.Organization.alias()
        AdvertiserOrganization = models.Organization.alias()
        AdDistributorContract = models.Contract.alias()
        PartnerContract = models.Contract.alias()
        AdvertiserContract = models.Contract.alias()
        AdDistributorAct = models.Act.alias()
        PartnerAct = models.Act.alias()

        query = db.select(
            [
                models.Client.id,
                models.Client.login,
                models.Client.client_id,
                models.Client.name,
                db.func.count(db.func.distinct(models.Campaign.id)).label('campaigns_count'),
                db.func.sum(models.ClientRow.suggested_amount).label('suggested_amount'),

                # ad_distributor = рекламная площадка или агентство для произвольного партнера
                db.func.bool_and(
                    and_(
                        AdDistributorAct.is_valid.is_(True),
                        AdDistributorContract.is_valid.is_(True),
                        AdDistributorOrganization.is_valid.is_(True),
                    )
                ).label('ad_distributor'),

                # ad_distributor_partner = агентство, произвольный партнер или клиент площадки
                db.func.bool_and(
                    AdDistributorPartnerOrganization.is_valid.is_(True),
                ).label('ad_distributor_partner'),

                # partner_client = контрагент агентства
                db.func.bool_and(
                    and_(
                        PartnerAct.is_valid.is_(True),
                        PartnerContract.is_valid.is_(True),
                        PartnerClientOrganization.is_valid.is_(True),
                    )
                ).label('partner_client'),

                # advertiser_contractor = контрагент конечного рекламодателя
                db.func.bool_and(
                    AdvertiserContractorOrganization.is_valid.is_(True)
                ).label('advertiser_contractor'),

                # advertiser = конечный рекламодатель
                db.func.bool_and(
                    and_(
                        AdvertiserContract.is_valid.is_(True),
                        AdvertiserOrganization.is_valid.is_(True),
                    )
                ).label('advertiser'),
            ]
        ).select_from(
            models.ClientRow
                .join(models.Client, models.Client.id == models.ClientRow.client_id)
                .join(models.Report, models.Report.id == models.Client.report_id)
                .outerjoin(models.Campaign, models.ClientRow.campaign_id == models.Campaign.id)
                .outerjoin(AdDistributorContract, models.ClientRow.ad_distributor_contract_id == AdDistributorContract.id)
                .outerjoin(AdDistributorOrganization, AdDistributorContract.contractor_id == AdDistributorOrganization.id)
                .outerjoin(AdDistributorPartnerOrganization, AdDistributorContract.client_id == AdDistributorPartnerOrganization.id)
                .outerjoin(AdDistributorAct, models.ClientRow.ad_distributor_act_id == AdDistributorAct.id)
                .outerjoin(PartnerContract, models.ClientRow.partner_contract_id == PartnerContract.id)
                .outerjoin(PartnerClientOrganization, PartnerContract.client_id == PartnerClientOrganization.id)
                .outerjoin(PartnerAct, models.ClientRow.partner_act_id == PartnerAct.id)
                .outerjoin(AdvertiserContract, models.ClientRow.advertiser_contract_id == AdvertiserContract.id)
                .outerjoin(AdvertiserOrganization, AdvertiserContract.client_id == AdvertiserOrganization.id)
                .outerjoin(AdvertiserContractorOrganization, AdvertiserContract.contractor_id == AdvertiserContractorOrganization.id)
        ).where(
            and_(
                models.Client.report_id == request.report_id,
                models.Report.agency_id == request.agency_id,
            )
        )

        if request.search_query:
            search_query = '%{}%'.format(request.search_query.lower())
            query = query.where(
                or_(
                    db.func.lower(models.Client.client_id).like(search_query),
                    db.func.lower(models.Client.login).like(search_query),
                    db.func.lower(models.Client.name).like(search_query),
                )
            )

        if request.is_valid is not None:
            operator = and_ if request.is_valid else or_
            query = query.where(
                operator(
                    AdDistributorContract.is_valid.is_(request.is_valid),
                    PartnerContract.is_valid.is_(request.is_valid),
                    AdvertiserContract.is_valid.is_(request.is_valid),
                    AdDistributorOrganization.is_valid.is_(request.is_valid),
                    AdDistributorPartnerOrganization.is_valid.is_(request.is_valid),
                    PartnerClientOrganization.is_valid.is_(request.is_valid),
                    AdvertiserContractorOrganization.is_valid.is_(request.is_valid),
                    AdvertiserOrganization.is_valid.is_(request.is_valid),
                    AdDistributorAct.is_valid.is_(request.is_valid),
                    PartnerAct.is_valid.is_(request.is_valid),
                )
            )

        if request.limit:
            query = query.limit(request.limit)

        if request.offset:
            query = query.offset(request.offset)

        rows = await query.group_by(models.Client.id, models.Client.login).order_by(models.Client.login).gino.all()

        return structs.ClientsInfoList(
            clients=[
                structs.ClientInfo(
                    id=row['id'],
                    client_id=row['client_id'],
                    login=row['login'],
                    name=row['name'],
                    suggested_amount=row['suggested_amount'],
                    campaigns_count=row['campaigns_count'],
                    has_valid_ad_distributor=row['ad_distributor'],
                    has_valid_ad_distributor_partner=row['ad_distributor_partner'],
                    has_valid_partner_client=row['partner_client'],
                    has_valid_advertiser=row['advertiser'],
                    has_valid_advertiser_contractor=row['advertiser_contractor'],
                ) for row in rows
            ],
            size=len(rows)
        )


@dataclass
class GetClientShortInfo:
    async def __call__(self, request: structs.ClientShortInfoInput) -> structs.ClientShortInfo:

        query = db.select(
            [
                models.Client.id,
                models.Client.client_id,
                models.Client.login,
                models.Client.name,
                models.Report.id.label('report_id'),
                models.Report.agency_id,
            ]
        ).select_from(
            models.Client.join(models.Report, models.Report.id == models.Client.report_id)
        ).where(
            models.Client.id == request.client_id,
        )

        res = await query.gino.first()

        if res is None:
            raise UnsuitableClientException()

        if res.report_id != request.report_id:
            raise UnsuitableReportException()

        if res.agency_id != request.agency_id:
            raise UnsuitableAgencyException()

        return structs.ClientShortInfo(
            id=res.id,
            client_id=res.client_id,
            login=res.login,
            name=res.name,
        )


@dataclass
class CreateClient:
    async def __call__(self, request: structs.CreateClientInput) -> structs.ClientInfo:
        report = await db.select([
            models.Report.id,
            models.Report.agency_id,
            models.ReportSettings.settings
        ]).select_from(models.Report.join(
            models.ReportSettings, models.Report.settings_id == models.ReportSettings.id
        )).where(
            models.Report.id == request.report_id
        ).gino.first()

        if not report:
            raise NoSuchReportException()

        if report.agency_id != request.agency_id:
            raise UnsuitableAgencyException()

        if report.settings.get('allow_create_clients') is False:
            raise ForbiddenByReportSettingsException()

        try:
            client = await models.Client.create(
                client_id=request.client_id,
                login=request.login,
                report_id=report.id,
                name=request.name,
                is_yt=False
            )

            return structs.ClientInfo(
                id=client.id,
                client_id=client.client_id,
                login=client.login,
                suggested_amount=None,
                campaigns_count=0,
                has_valid_ad_distributor=False,
                has_valid_ad_distributor_partner=False,
                has_valid_partner_client=False,
                has_valid_advertiser_contractor=False,
                has_valid_advertiser=False,
            )

        except asyncpg.exceptions.UniqueViolationError:
            raise UniqueViolationClientException()
