import asyncio
from dataclasses import dataclass
from sqlalchemy import or_
from crm.agency_cabinet.ord.server.src.db import db, models
from crm.agency_cabinet.ord.common import structs


@dataclass
class GetContracts:
    async def __call__(self, request: structs.GetContractsInput) -> structs.ContractsList:
        client = models.Organization.alias()
        contractor = models.Organization.alias()
        query = db.select(
            [
                models.Contract.id.label('contract_id'),
                models.Contract.contract_eid,
                models.Contract.is_reg_report,
                models.Contract.type,
                models.Contract.action_type,
                models.Contract.subject_type,
                models.Contract.date,
                models.Contract.amount,
                models.Contract.is_vat,


                client.id.label('client_organization_id'),
                client.type.label('client_type'),
                client.name.label('client_name'),
                client.inn.label('client_inn'),
                client.is_rr.label('client_is_rr'),
                client.is_ors.label('client_is_ors'),
                client.mobile_phone.label('client_mobile_phone'),
                client.epay_number.label('client_epay_number'),
                client.reg_number.label('client_reg_number'),
                client.alter_inn.label('client_alter_inn'),
                client.oksm_number.label('client_oksm_number'),
                client.rs_url.label('client_rs_url'),

                contractor.id.label('contractor_organization_id'),
                contractor.type.label('contractor_type'),
                contractor.name.label('contractor_name'),
                contractor.inn.label('contractor_inn'),
                contractor.is_rr.label('contractor_is_rr'),
                contractor.is_ors.label('contractor_is_ors'),
                contractor.mobile_phone.label('contractor_mobile_phone'),
                contractor.epay_number.label('contractor_epay_number'),
                contractor.reg_number.label('contractor_reg_number'),
                contractor.alter_inn.label('contractor_alter_inn'),
                contractor.oksm_number.label('contractor_oksm_number'),
                contractor.rs_url.label('contractor_rs_url'),
            ]
        ).select_from(
            models.Contract.join(client, client.id == models.Contract.client_id).join(
                contractor, contractor.id == models.Contract.contractor_id)
        ).where(or_(client.partner_id == request.agency_id, contractor.partner_id == request.agency_id))

        size_query = db.select(
            [db.func.count()]
        ).select_from(
            models.Contract.join(client, client.id == models.Contract.client_id).join(
                contractor, contractor.id == models.Contract.contractor_id)
        ).where(or_(client.partner_id == request.agency_id, contractor.partner_id == request.agency_id))

        if request.search_query:
            search_query = '%{}%'.format(request.search_query.lower())
            condition = db.func.lower(models.Contract.contract_eid).like(search_query)

            query = query.where(condition)
            size_query = size_query.where(condition)

        get_size_task = asyncio.create_task(size_query.gino.scalar())

        if request.limit:
            query = query.limit(request.limit)

        if request.offset:
            query = query.offset(request.offset)

        contracts = await query.order_by(models.Contract.id.desc()).gino.all()

        return structs.ContractsList(
            size=await get_size_task,
            contracts=[structs.Contract(
                id=contract.contract_id,
                contract_eid=contract.contract_eid,
                is_reg_report=contract.is_reg_report,
                type=contract.type,
                action_type=contract.action_type,
                subject_type=contract.subject_type,
                date=contract.date,
                amount=contract.amount,
                is_vat=contract.is_vat,
                client_organization=structs.Organization(
                    id=contract.client_organization_id,
                    type=contract.client_type,
                    name=contract.client_name,
                    inn=contract.client_inn,
                    is_rr=contract.client_is_rr,
                    is_ors=contract.client_is_ors,
                    mobile_phone=contract.client_mobile_phone,
                    epay_number=contract.client_epay_number,
                    reg_number=contract.client_reg_number,
                    alter_inn=contract.client_alter_inn,
                    oksm_number=contract.client_oksm_number,
                    rs_url=contract.client_rs_url
                ) if contract.client_organization_id is not None else None,
                contractor_organization=structs.Organization(
                    id=contract.contractor_organization_id,
                    type=contract.contractor_type,
                    name=contract.contractor_name,
                    inn=contract.contractor_inn,
                    is_rr=contract.contractor_is_rr,
                    is_ors=contract.contractor_is_ors,
                    mobile_phone=contract.contractor_mobile_phone,
                    epay_number=contract.contractor_epay_number,
                    reg_number=contract.contractor_reg_number,
                    alter_inn=contract.contractor_alter_inn,
                    oksm_number=contract.contractor_oksm_number,
                    rs_url=contract.contractor_rs_url
                ) if contract.contractor_organization_id is not None else None,
            ) for contract in contracts]
        )
