from dataclasses import dataclass
from sqlalchemy import and_

from crm.agency_cabinet.common.server.common.structs import TaskStatuses
from crm.agency_cabinet.ord.common import structs
from crm.agency_cabinet.ord.server.src.db import models, db
from crm.agency_cabinet.ord.common.exceptions import NoSuchReportException, UnsuitableAgencyException, ImportDataAlreadyRequested


@dataclass
class ImportData:
    async def __call__(self, request: structs.ImportDataInput) -> structs.TaskInfo:
        report_import_info = await db.select([
            models.Report.id,
            models.Report.agency_id,
            models.ReportImportInfo.id.label('report_import_id'),
            models.ReportImportInfo.status,
        ]).select_from(
            models.Report.outerjoin(models.ReportImportInfo)
        ).where(
            models.Report.id == request.report_id
        ).gino.first()

        if not report_import_info:
            raise NoSuchReportException()

        if report_import_info.agency_id != request.agency_id:
            raise UnsuitableAgencyException()

        if report_import_info.status is not None and report_import_info.status in (TaskStatuses.in_progress.value, TaskStatuses.requested.value):
            raise ImportDataAlreadyRequested()

        if request.display_name is None:
            display_name = request.filename.split('/')[-1]
            if len(display_name) > 37:
                display_name = display_name[37:]
        else:
            display_name = request.display_name

        file = await models.S3MdsFile.create(
            bucket=request.bucket,
            name=request.filename,
            display_name=display_name
        )

        import_task = await models.ReportImportInfo.create(
            report_id=request.report_id,
            file_id=file.id,
            status=TaskStatuses.requested.value
        )

        # import_report_task.delay(import_task.id)

        return structs.TaskInfo(
            task_id=import_task.id,
            status=TaskStatuses(import_task.status)
        )


@dataclass
class GetLockStatus:
    async def __call__(self, request: structs.GetLockStatusInput) -> structs.LockStatus:
        report = await models.Report.query.where(models.Report.id == request.report_id).gino.first()

        if not report:
            raise NoSuchReportException()

        if report.agency_id != request.agency_id:
            raise UnsuitableAgencyException()

        task = await models.ReportImportInfo.query.where(
            and_(
                models.ReportImportInfo.report_id == request.report_id,
                models.ReportImportInfo.status.in_([TaskStatuses.requested.value, TaskStatuses.in_progress.value])
            )
        ).gino.first()

        return structs.LockStatus(
            lock=task is not None
        )
