import asyncio

from dataclasses import dataclass
from sqlalchemy import or_

from crm.agency_cabinet.ord.server.src.db import db, models
from crm.agency_cabinet.ord.common import structs, consts


@dataclass
class GetOrganizations:
    async def __call__(self, request: structs.GetOrganizationsInput) -> structs.OrganizationsList:
        query = db.select([
            models.Organization
        ]).select_from(models.Organization).where(
            models.Organization.partner_id == request.partner_id
        )

        size_query = db.select(
            [db.func.count()]
        ).select_from(models.Organization).where(
            models.Organization.partner_id == request.partner_id
        )

        if request.search_query:
            search_query = '%{}%'.format(request.search_query.lower())

            condition = or_(
                db.func.lower(models.Organization.name).like(search_query),
                db.func.lower(models.Organization.inn).like(search_query),
                db.func.lower(models.Organization.mobile_phone).like(search_query),
            )

            query = query.where(condition)
            size_query = size_query.where(condition)

        get_size_task = asyncio.create_task(size_query.gino.scalar())

        if request.offset:
            query = query.offset(request.offset)

        if request.limit:
            query = query.limit(request.limit)

        organizations = await query.order_by(models.Organization.id.desc()).gino.all()

        return structs.OrganizationsList(
            size=await get_size_task,
            organizations=[structs.Organization(
                id=org.id,
                type=consts.OrganizationType(org.type) if org.type else None,
                name=org.name,
                inn=org.inn,
                is_rr=org.is_rr,
                is_ors=org.is_ors,
                mobile_phone=org.mobile_phone,
                epay_number=org.epay_number,
                reg_number=org.reg_number,
                alter_inn=org.alter_inn,
                oksm_number=org.oksm_number,
                rs_url=org.rs_url,
            ) for org in organizations]
        )
