import logging
from dataclasses import dataclass
from sqlalchemy import and_
from datetime import datetime, timezone

from crm.agency_cabinet.common.server.common.structs import TaskStatuses
from crm.agency_cabinet.common.mds import create_presigned_url_async
from crm.agency_cabinet.common.mds.definitions import S3AsyncClient

from crm.agency_cabinet.ord.server.src.db import Report
from crm.agency_cabinet.ord.common import structs, consts
from crm.agency_cabinet.ord.server.src.db.queries import build_get_reports, build_get_client_rows_query
from crm.agency_cabinet.ord.server.src.db import models, db
from crm.agency_cabinet.ord.common.exceptions import NoSuchReportException, UnsuitableAgencyException, \
    ReportAlreadySentException, ReportNotReadyException, FileNotFoundException

LOGGER = logging.getLogger('ord_report_procedure')


def get_report_info_struct(report: models.Report, report_settings: models.ReportSettings, clients_count: int, campaigns_count: int):
    return structs.ReportInfo(
        report_id=report.id,
        status=consts.ReportStatuses(report.status),
        period_from=report.period_from,
        reporter_type=consts.ReporterType(report.reporter_type),
        clients_count=clients_count,
        campaigns_count=campaigns_count,
        sending_date=report.sending_date,
        settings=structs.ReportSettings(
            name=report_settings.name,
            display_name=report_settings.display_name,
            allow_create_ad_distributor_acts=report_settings.allow_create_ad_distributor_acts,
            allow_create_clients=report_settings.allow_create_clients,
            allow_create_campaigns=report_settings.allow_create_campaigns,
            allow_edit_report=report_settings.allow_edit_report,
        )
    )


@dataclass
class GetReportsInfo:
    async def __call__(self, request: structs.GetReportsInfoRequest) -> structs.GetReportsInfoResponse:
        res = await build_get_reports(
            agency_id=request.agency_id,
            search_query=request.search_query,
            period_from=request.period_from,
            period_to=request.period_to,
            status=request.status,
            sort=request.sort,
            limit=request.limit,
            offset=request.offset
        ).all()

        return structs.GetReportsInfoResponse(
            reports=[
                get_report_info_struct(report, report_settings, clients_count, campaigns_count)
                for (report, report_settings, clients_count, campaigns_count) in res
            ]
        )


@dataclass
class GetDetailedReportInfo:
    async def __call__(self, request: structs.GetDetailedReportInfoRequest) -> structs.ReportInfo:
        result = await build_get_reports(
            report_id=request.report_id,
        ).first()

        if result is None:
            raise NoSuchReportException(f'Can\'t find report with id {request.report_id}')

        report, report_settings, clients_count, campaigns_count = result
        if report.agency_id != request.agency_id:
            raise UnsuitableAgencyException(f'Unsuitable agency for report with id {request.report_id}')

        return get_report_info_struct(report, report_settings, clients_count, campaigns_count)


@dataclass
class SendReport:

    async def __call__(self, request: structs.SendReportInput):
        report = await Report.query.where(Report.id == request.report_id).gino.first()

        if report is None:
            raise NoSuchReportException(f'Can\'t find report with id {request.report_id}')

        if report.agency_id != request.agency_id:
            raise UnsuitableAgencyException(f'Unsuitable agency for report with id {request.report_id}')

        if report.status == consts.ReportStatuses.sent.value:
            raise ReportAlreadySentException(f'Report with id {request.report_id} was already sent')

        await Report.update.values(
            status=consts.ReportStatuses.sent.value,
            sending_date=datetime.now(tz=timezone.utc)
        ).where(Report.id == request.report_id).gino.status(
            read_only=False, reuse=False)


@dataclass
class ReportExport:

    async def __call__(self, request: structs.ReportExportRequest):
        report_export_info = await db.select([
            models.Report.agency_id,
            models.ReportExportInfo.id,
            models.ReportExportInfo.status,
            models.ReportExportInfo.created_at.label('exported_at'),
        ]).select_from(
            models.Report.outerjoin(models.ReportExportInfo)
        ).where(
            models.Report.id == request.report_id
        ).order_by(
            models.ReportExportInfo.created_at
        ).gino.first()

        if not report_export_info:
            raise NoSuchReportException(f'Can\'t find report with id {request.report_id}')

        if report_export_info['agency_id'] != request.agency_id:
            raise UnsuitableAgencyException(f'Unsuitable agency for report with id {request.report_id}')

        last_export_task = report_export_info['id'] is not None

        create_new_task = not last_export_task

        if last_export_task:
            query = build_get_client_rows_query(request.report_id, only_max_update=True)
            report_data = await query.gino.first()
            create_new_task = False
            if report_data and report_data['updated_at'] and report_data['updated_at'] > report_export_info['exported_at']:
                create_new_task = True

        if create_new_task:
            report_export_info = await models.ReportExportInfo.create(
                report_id=request.report_id,
                status=TaskStatuses.requested.value,
            )

        return structs.ReportExportResponse(
            report_export_id=report_export_info.id,
            status=TaskStatuses(report_export_info.status)
        )


@dataclass
class GetReportExportInfo:

    async def __call__(self, request: structs.ReportExportInfoRequest) -> structs.ReportExportResponse:
        report_export_info = await db.select([
            models.Report.id,
            models.Report.agency_id,
            models.ReportExportInfo.id.label('report_export_id'),
            models.ReportExportInfo.status,
        ]).select_from(
            models.Report.outerjoin(models.ReportExportInfo)
        ).where(
            and_(
                models.ReportExportInfo.id == request.report_export_id,
                models.ReportExportInfo.report_id == request.report_id,
            )
        ).gino.first()

        if report_export_info is None:
            raise NoSuchReportException
        if report_export_info['agency_id'] != request.agency_id:
            raise UnsuitableAgencyException(f'Unsuitable agency for report with id {request.report_id}')

        return structs.ReportExportResponse(
            report_export_id=report_export_info.report_export_id,
            status=TaskStatuses(report_export_info.status)
        )


@dataclass
class GetReportUrl:
    async def __call__(self, request: structs.GetReportUrlRequest,
                       s3_client: S3AsyncClient) -> structs.GetReportUrlResponse:

        report_mds_data = await db.select([
            models.Report.agency_id,
            models.ReportExportInfo.status,
            models.ReportExportInfo.file_id,
            models.S3MdsFile.bucket,
            models.S3MdsFile.name
        ]).select_from(
            models.Report.join(
                models.ReportExportInfo).outerjoin(
                models.S3MdsFile, models.ReportExportInfo.file_id == models.S3MdsFile.id)
        ).where(
            and_(
                models.Report.id == request.report_id,
                models.ReportExportInfo.id == request.report_export_id,
            )
        ).gino.first()

        if report_mds_data is None:
            raise NoSuchReportException
        if report_mds_data['agency_id'] != request.agency_id:
            raise UnsuitableAgencyException
        if report_mds_data['status'] != TaskStatuses.ready.value:
            raise ReportNotReadyException
        if report_mds_data['file_id'] is None:
            raise FileNotFoundException

        url = await create_presigned_url_async(s3_client, report_mds_data['bucket'], report_mds_data['name'])

        return structs.GetReportUrlResponse(report_url=url)


@dataclass
class DeleteReport:
    async def __call__(self, request: structs.DeleteReportRequest):
        report = await models.Report.query.where(
            models.Report.id == request.report_id
        ).gino.first()

        if not report:
            raise NoSuchReportException()

        if report.agency_id != request.agency_id:
            raise UnsuitableAgencyException()

        return await models.Report.update.values(is_deleted=True).where(
            models.Report.id == request.report_id
        ).gino.status(read_only=False, reuse=False)


class CreateReport:
    async def __call__(self, request: structs.CreateReportRequest):
        report_settings = await models.ReportSettings.query.where(
            models.ReportSettings.name == 'other'
        ).gino.first()

        new_report = await models.Report.create(
            agency_id=request.agency_id,
            settings_id=report_settings.id,
            period_from=request.period_from,
            status=consts.ReportStatuses.draft.value,
            reporter_type=request.reporter_type.value
        )

        report, report_settings, clients_count, campaigns_count = await build_get_reports(report_id=new_report.id).first()
        return get_report_info_struct(report, report_settings, clients_count, campaigns_count)
