import logging
import typing

from datetime import datetime

from google.protobuf.wrappers_pb2 import BoolValue
from crm.agency_cabinet.common.enum import get_enum_from_str
from crm.agency_cabinet.common.client import BaseClient
from crm.agency_cabinet.common.client.exceptions import ProtocolError
from crm.agency_cabinet.rewards.common import QUEUE, structs
from crm.agency_cabinet.rewards.proto import (
    common_pb2, request_pb2, rewards_info_pb2, contracts_info_pb2, calculator_pb2, reports_pb2, dashboard_pb2, documents_pb2
)
from crm.agency_cabinet.common.consts.report import ReportsTypes
from crm.agency_cabinet.common.consts.calculator import CalculatorServiceType
from crm.agency_cabinet.common.consts.service import Services
from crm.agency_cabinet.common.proto_utils import timestamp_or_none

from .exceptions import NoSuchReportException, UnsuitableAgency, NoSuchRewardException, \
    UnknownClientsException, UnknownReportTypeException, UnknownServiceException, BunkerError, NoSuchContractException, \
    FileNotFoundException, ReportNotReadyException, UnsupportedReportParametersException, NoSuchDocumentException, \
    NoSuchCalculatorDataException, UnsupportedParametersException, UnknownCalculatorServiceTypeException, \
    BunkerNotFoundException


LOGGER = logging.getLogger('rewards.client')


class RewardsClient(BaseClient):
    queue = QUEUE

    async def ping(self) -> str:
        _, data = await self._send_message(
            request_pb2.RpcRequest(ping=common_pb2.Empty()),
            common_pb2.PingOutput
        )
        return str(data)

    async def get_rewards_info(self, agency_id: int,
                               filter_from: datetime = None,
                               filter_to: datetime = None,
                               filter_contract: int = None,
                               filter_type: str = None,
                               filter_is_paid: bool = None) -> list[structs.RewardInfo]:
        _filter_from = timestamp_or_none(filter_from)
        _filter_to = timestamp_or_none(filter_to)

        request = request_pb2.RpcRequest(
            get_rewards_info=rewards_info_pb2.GetRewardsInfo(
                agency_id=agency_id,
                filter_from=_filter_from,
                filter_to=_filter_to,
                filter_contract=filter_contract,
                filter_type=filter_type,
                filter_is_paid=BoolValue(
                    value=filter_is_paid) if filter_is_paid is not None else None))

        result, data = await self._send_message(
            request,
            rewards_info_pb2.GetRewardsInfoOutput
        )

        if result == 'result':
            return structs.GetRewardsInfoResponse.from_proto(data).rewards

        raise ProtocolError('Unexpected response')

    async def get_dashboard(
        self, agency_id: int, year: int, filter_contract: int = None, filter_service: str = None
    ) -> list[structs.DashboardItem]:
        request = request_pb2.RpcRequest(
            get_dashboard=dashboard_pb2.GetDashboard(
                agency_id=agency_id,
                filter_contract=filter_contract,
                filter_service=filter_service,
                year=year
            )
        )

        result, data = await self._send_message(request, dashboard_pb2.DashboardOutput)

        if result == 'result':
            return structs.GetDashboardResponse.from_proto(data).dashboard

        if result == 'unsupported_parameters':
            raise UnsupportedParametersException()

        raise ProtocolError('Unexpected response')

    async def get_contracts_info(self, agency_id: int) -> list[structs.ContractInfo]:
        request = request_pb2.RpcRequest(
            get_contracts_info=contracts_info_pb2.GetContractsInfo(agency_id=agency_id)
        )

        result, data = await self._send_message(
            request,
            contracts_info_pb2.GetContractsInfoOutput
        )

        if result == 'result':
            return structs.GetContractsInfoResponse.from_proto(data).contracts

        raise ProtocolError('Unexpected response')

    async def get_calculator_meta(self, agency_id: int, contract_id: int,
                                  service: typing.Union[str, CalculatorServiceType], version: str) -> structs.GetCalculatorMetaResponse:

        request = request_pb2.RpcRequest(
            get_calculator_meta=structs.GetCalculatorMetaRequest(
                agency_id=agency_id,
                contract_id=contract_id,
                service=get_enum_from_str(service, CalculatorServiceType, UnknownCalculatorServiceTypeException),
                version=version,
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            calculator_pb2.GetCalculatorMetaOutput
        )

        if result == 'result':
            return structs.GetCalculatorMetaResponse.from_proto(data)

        if result == 'not_found':
            raise BunkerNotFoundException()

        if result == 'no_such_contract':
            raise NoSuchContractException()

        if result == 'bunker_error':
            raise BunkerError(data)

        raise ProtocolError('Unexpected response')

    async def get_calculator_data(self, agency_id: int, contract_id: int,
                                  service: typing.Union[str, CalculatorServiceType], version: str) -> structs.GetCalculatorDataResponse:
        request = request_pb2.RpcRequest(
            get_calculator_data=structs.GetCalculatorDataRequest(
                agency_id=agency_id,
                contract_id=contract_id,
                service=get_enum_from_str(service, CalculatorServiceType, UnknownCalculatorServiceTypeException),
                version=version
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            calculator_pb2.GetCalculatorDataOutput
        )

        if result == 'result':
            return structs.GetCalculatorDataResponse.from_proto(data)

        if result == 'not_found':
            raise NoSuchCalculatorDataException()

        raise ProtocolError('Unexpected response')

    async def get_detailed_reward_info(self, agency_id: int, reward_id: int) -> structs.DetailedRewardInfo:
        request = request_pb2.RpcRequest(
            get_detailed_reward_info=rewards_info_pb2.GetDetailedRewardInfo(
                agency_id=agency_id,
                reward_id=reward_id,
            )
        )

        result, data = await self._send_message(
            request,
            rewards_info_pb2.GetDetailedRewardInfoOutput
        )

        if result == 'result':
            return structs.GetDetailedRewardInfoResponse.from_proto(data).reward

        if result == 'no_such_reward':
            LOGGER.debug(f'There is no reward with id equal to {reward_id}')
            raise NoSuchRewardException()

        if result == 'unsuitable_agency':
            LOGGER.debug(f'Agency with id equal to {agency_id} don\'t have a reward with id equal to {reward_id}')
            raise UnsuitableAgency()

        raise ProtocolError('Unexpected response')

    async def get_reports_info(self, agency_id: int,
                               filter_contract: int = None,
                               filter_type: str = None,
                               filter_service: str = None) -> typing.List[structs.ReportInfo]:
        request = request_pb2.RpcRequest(
            get_reports_info=reports_pb2.GetReportsInfo(
                agency_id=agency_id,
                filter_contract=filter_contract,
                filter_service=filter_service,
                filter_type=filter_type
            )
        )

        result, data = await self._send_message(
            request,
            reports_pb2.GetReportsInfoOutput
        )

        if result == 'result':
            return structs.GetReportsInfoResponse.from_proto(data).reports

        raise ProtocolError('Unexpected response')

    async def get_detailed_report_info(self, agency_id: int, report_id: int) -> structs.DetailedReportInfo:
        request = request_pb2.RpcRequest(
            get_detailed_report_info=reports_pb2.GetDetailedReportInfo(
                agency_id=agency_id,
                report_id=report_id
            )
        )

        result, data = await self._send_message(
            request,
            reports_pb2.GetDetailedReportInfoOutput
        )

        if result == 'result':
            return structs.GetDetailedReportInfoResponse.from_proto(data).report
        elif result == 'no_such_report':
            raise NoSuchReportException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgency()
        raise ProtocolError('Unexpected response')

    async def delete_report(self, agency_id: int, report_id: int) -> bool:
        request = request_pb2.RpcRequest(
            delete_report=reports_pb2.DeleteReport(
                agency_id=agency_id,
                report_id=report_id
            )
        )

        result, data = await self._send_message(
            request,
            reports_pb2.DeleteReportOutput
        )
        if result == 'result':
            return structs.DeleteReportResponse.from_proto(data).is_deleted
        elif result == 'no_such_report':
            raise NoSuchReportException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgency()
        raise ProtocolError('Unexpected response')

    async def create_report(self, agency_id: int, contract_id: int, name: str, type: str, service: str,
                            period_from: datetime, period_to: datetime, clients_ids: list = ()) -> structs.DetailedReportInfo:

        if not Services.has(service):
            raise UnknownServiceException()
        elif not ReportsTypes.has(type):
            raise UnknownReportTypeException()  # TODO: use enum

        request = request_pb2.RpcRequest(
            create_report=reports_pb2.CreateReport(
                agency_id=agency_id,
                contract_id=contract_id,
                name=name,
                type=type,
                service=service,
                period_from=timestamp_or_none(period_from),
                period_to=timestamp_or_none(period_to),
                clients_ids=clients_ids
            )
        )

        result, data = await self._send_message(
            request,
            reports_pb2.CreateReportOutput
        )

        if result == 'result':
            return structs.CreateReportResponse.from_proto(data).report
        elif result == 'unsuitable_agency':
            raise UnsuitableAgency()
        elif result == 'no_such_contract':
            raise NoSuchContractException()
        elif result == 'unknown_clients':
            raise UnknownClientsException()
        elif result == 'unsupported_parameters':
            raise UnsupportedReportParametersException(data)
        raise ProtocolError('Unexpected response')

    async def get_report_url(self, agency_id: int, report_id: int) -> str:
        request = request_pb2.RpcRequest(
            get_report_url=reports_pb2.GetReportUrl(
                agency_id=agency_id,
                report_id=report_id
            )
        )

        result, data = await self._send_message(
            request,
            reports_pb2.GetReportUrlOutput
        )
        if result == 'result':
            return structs.GetReportUrlResponse.from_proto(data).report_url
        elif result == 'no_such_report':
            raise NoSuchReportException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgency()
        elif result == 'file_not_found':
            raise FileNotFoundException()
        elif result == 'report_not_ready':
            raise ReportNotReadyException()
        raise ProtocolError('Unexpected response')

    async def get_document_url(self, agency_id: int, document_id: int) -> str:
        request = request_pb2.RpcRequest(
            get_document_url=documents_pb2.GetDocumentUrl(
                agency_id=agency_id,
                document_id=document_id
            )
        )

        result, data = await self._send_message(
            request,
            documents_pb2.GetDocumentUrlOutput
        )
        if result == 'result':
            return structs.GetDocumentUrlResponse.from_proto(data).document_url
        elif result == 'no_such_document':
            raise NoSuchDocumentException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgency()
        elif result == 'file_not_found':
            raise FileNotFoundException()
        raise ProtocolError('Unexpected response')

    async def list_available_calculators(self, agency_id: int, contract_id: int = None) -> structs.ListAvailableCalculatorsResponse:
        request = request_pb2.RpcRequest(
            list_available_calculators=calculator_pb2.ListAvailableCalculatorsInput(
                agency_id=agency_id,
                contract_id=contract_id
            )
        )
        result, data = await self._send_message(
            request,
            calculator_pb2.ListAvailableCalculatorsOutput
        )
        if result == 'result':
            return structs.ListAvailableCalculatorsResponse.from_proto(data)
        raise ProtocolError('Unexpected response')
