import typing

from dataclasses import dataclass
from datetime import datetime
from decimal import Decimal
from crm.agency_cabinet.common.consts import ContractType, CalculatorServiceType
from crm.agency_cabinet.common.proto_utils import BaseStruct
from crm.agency_cabinet.rewards.proto import calculator_pb2
from crm.agency_cabinet.rewards.common.structs.common import contract_type_converter, calculator_service_type_converter


@dataclass
class GetCalculatorMetaRequest(BaseStruct):
    agency_id: int
    contract_id: int
    service: CalculatorServiceType
    version: str
    # contract_type: ContractType

    @classmethod
    def from_proto(cls, message: calculator_pb2.GetCalculatorMeta) -> 'GetCalculatorMetaRequest':
        return cls(
            agency_id=message.agency_id,
            contract_id=message.contract_id,
            service=calculator_service_type_converter.forward(message.service),
            version=message.version,
            # contract_type=contract_type_converter.forward(message.contract_type)
        )

    def to_proto(self) -> calculator_pb2.GetCalculatorMeta:
        return calculator_pb2.GetCalculatorMeta(
            agency_id=self.agency_id,
            contract_id=self.contract_id,
            service=calculator_service_type_converter.reversed(self.service),
            version=self.version,
            # contract_type=contract_type_converter.reversed(self.contract_type)
        )


@dataclass
class GetCalculatorMetaResponse(BaseStruct):
    result: str

    @classmethod
    def from_proto(cls, result: str) -> 'GetCalculatorMetaResponse':
        return cls(result=result)

    def to_proto(self) -> calculator_pb2.GetCalculatorMetaOutput:
        return calculator_pb2.GetCalculatorMetaOutput(result=self.result)


@dataclass
class GetCalculatorDataRequest(BaseStruct):
    contract_id: int
    agency_id: int
    service: CalculatorServiceType
    version: str

    @classmethod
    def from_proto(cls, message: calculator_pb2.GetCalculatorData) -> 'GetCalculatorDataRequest':
        return cls(
            agency_id=message.agency_id,
            contract_id=message.contract_id,
            service=calculator_service_type_converter.forward(message.service),
            version=message.version
        )

    def to_proto(self) -> calculator_pb2.GetCalculatorData:
        return calculator_pb2.GetCalculatorData(
            agency_id=self.agency_id,
            contract_id=self.contract_id,
            service=calculator_service_type_converter.reversed(self.service),
            version=self.version
        )


@dataclass
class GetCalculatorDataResponse(BaseStruct):
    result: str

    @classmethod
    def from_proto(cls, result: str) -> 'GetCalculatorDataResponse':
        return cls(result=result)

    def to_proto(self) -> calculator_pb2.GetCalculatorMetaOutput:
        return calculator_pb2.GetCalculatorDataOutput(result=self.result)


@dataclass
class CalculatorGradeData(BaseStruct):
    grade_id: str
    domains_count: int = None
    revenue_average: Decimal = None
    revenue: Decimal = None


@dataclass
class CalculatorIndexData:
    index_id: str
    revenue: Decimal


@dataclass
class CalculatorMonthData:
    indexes: list[CalculatorIndexData]
    predict: bool
    period_from: datetime
    grades: list[CalculatorGradeData] = None


@dataclass
class CalculatorData:
    months: list[CalculatorMonthData]


@dataclass
class ListAvailableCalculatorsInput(BaseStruct):
    agency_id: int
    contract_id: typing.Optional[int] = None

    @classmethod
    def from_proto(cls, message: calculator_pb2.ListAvailableCalculatorsInput) -> 'ListAvailableCalculatorsInput':
        return cls(agency_id=message.agency_id, contract_id=message.contract_id if message.HasField('contract_id') else None)

    def to_proto(self) -> calculator_pb2.ListAvailableCalculatorsInput:
        return calculator_pb2.ListAvailableCalculatorsInput(agency_id=self.agency_id, contract_id=self.contract_id)


@dataclass
class CalculatorDescription(BaseStruct):
    contract_id: int
    service: CalculatorServiceType
    version: str
    contract_type: ContractType

    @classmethod
    def from_proto(cls, message: calculator_pb2.CalculatorDescription) -> 'CalculatorDescription':
        return cls(
            contract_id=message.contract_id,
            service=calculator_service_type_converter.forward(message.service),
            version=message.version,
            contract_type=contract_type_converter.forward(message.contract_type)
        )

    def to_proto(self) -> calculator_pb2.CalculatorDescription:
        return calculator_pb2.CalculatorDescription(
            contract_id=self.contract_id,
            service=calculator_service_type_converter.reversed(self.service),
            version=self.version,
            contract_type=contract_type_converter.reversed(self.contract_type)
        )


@dataclass
class ListAvailableCalculatorsResponse(BaseStruct):
    calculators_descriptions: typing.List[CalculatorDescription]

    @classmethod
    def from_proto(cls, message: calculator_pb2.CalculatorDescriptionList) -> 'ListAvailableCalculatorsResponse':
        return cls(
            calculators_descriptions=[CalculatorDescription.from_proto(description) for description in message.calculators_descriptions]
        )

    def to_proto(self) -> calculator_pb2.CalculatorDescriptionList:
        return calculator_pb2.CalculatorDescriptionList(
            calculators_descriptions=[description.to_proto() for description in self.calculators_descriptions]
        )
