from dataclasses import dataclass
from datetime import datetime

from crm.agency_cabinet.common.consts import ContractType
from crm.agency_cabinet.common.proto_utils import BaseStruct, timestamp_or_none
from crm.agency_cabinet.rewards.proto import contracts_info_pb2
from crm.agency_cabinet.rewards.common.structs.common import contract_type_converter


@dataclass
class GetContractsInfoRequest(BaseStruct):
    agency_id: int

    @classmethod
    def from_proto(cls, message: contracts_info_pb2.GetContractsInfo) -> 'GetContractsInfoRequest':
        return cls(agency_id=message.agency_id)


@dataclass
class ContractInfo(BaseStruct):
    contract_id: int
    eid: str
    inn: str
    payment_type: str
    services: list[str]  # TODO: use enum list
    finish_date: datetime
    type: ContractType
    is_crisis: bool

    @classmethod
    def from_proto(cls, message: contracts_info_pb2.ContractInfo) -> 'ContractInfo':
        return cls(
            contract_id=message.contract_id,
            eid=message.eid,
            inn=message.inn,
            payment_type=message.payment_type,
            services=message.services,
            finish_date=message.finish_date.ToDatetime(),
            type=contract_type_converter.forward(message.type),
            is_crisis=message.is_crisis,
        )

    def to_proto(self) -> contracts_info_pb2.ContractInfo:
        finish_date = timestamp_or_none(self.finish_date)

        return contracts_info_pb2.ContractInfo(
            contract_id=self.contract_id,
            eid=self.eid,
            inn=self.inn,
            payment_type=self.payment_type,
            services=self.services,
            finish_date=finish_date,
            type=contract_type_converter.reversed(self.type),
            is_crisis=self.is_crisis,
        )


@dataclass
class GetContractsInfoResponse(BaseStruct):
    contracts: list[ContractInfo]

    @classmethod
    def from_proto(cls, message: contracts_info_pb2.ContractsInfoList) -> 'GetContractsInfoResponse':
        return cls(contracts=[ContractInfo.from_proto(contract) for contract in message.contracts])

    def to_proto(self) -> contracts_info_pb2.ContractsInfoList:
        return contracts_info_pb2.ContractsInfoList(contracts=[contract.to_proto() for contract in self.contracts])
