import typing

from dataclasses import dataclass
from datetime import datetime
from decimal import Decimal

from crm.agency_cabinet.common.consts import Services
from crm.agency_cabinet.common.proto_utils import (
    BaseStruct, timestamp_or_none, decimal_percent_to_string, decimal_to_string
)
from crm.agency_cabinet.rewards.proto import dashboard_pb2


@dataclass
class GetDashboard(BaseStruct):
    agency_id: int
    filter_contract: int
    filter_service: str  # TODO: rename, use enum
    year: int

    @classmethod
    def from_proto(cls, message: dashboard_pb2.GetDashboard) -> 'GetDashboard':
        return cls(
            agency_id=message.agency_id,
            filter_contract=message.filter_contract,
            filter_service=message.filter_service,
            year=message.year
        )


@dataclass
class DashboardReward(BaseStruct):
    reward: Decimal
    reward_percent: Decimal
    predict: bool
    period_from: datetime

    @classmethod
    def from_proto(cls, message: dashboard_pb2.DashboardReward) -> 'DashboardReward':
        return cls(
            reward=Decimal(message.reward),
            reward_percent=Decimal(message.reward_percent),
            predict=message.predict,
            period_from=message.period_from.ToDatetime()
        )

    def to_proto(self) -> dashboard_pb2.DashboardReward:
        return dashboard_pb2.DashboardReward(
            reward=decimal_to_string(self.reward.normalize()),
            reward_percent=decimal_percent_to_string(self.reward_percent.normalize()),
            predict=self.predict,
            period_from=timestamp_or_none(self.period_from)
        )


@dataclass
class DashboardRewardsMap(BaseStruct):
    month: list[DashboardReward]
    quarter: list[DashboardReward]
    semiyear: list[DashboardReward]

    @classmethod
    def from_proto(cls, message: dashboard_pb2.DashboardRewardsMap) -> 'DashboardRewardsMap':
        return cls(
            month=[DashboardReward.from_proto(month_reward) for month_reward in message.month],
            quarter=[DashboardReward.from_proto(quarter_reward) for quarter_reward in message.quarter],
            semiyear=[DashboardReward.from_proto(semiyear_reward) for semiyear_reward in message.semiyear]
        )

    def to_proto(self) -> dashboard_pb2.DashboardRewardsMap:
        return dashboard_pb2.DashboardRewardsMap(
            month=[month_reward.to_proto() for month_reward in self.month],
            quarter=[quarter_reward.to_proto() for quarter_reward in self.quarter],
            semiyear=[semiyear_reward.to_proto() for semiyear_reward in self.semiyear]
        )


@dataclass
class DashboardItem(BaseStruct):
    contract_id: int
    service: Services
    active: bool
    rewards: typing.Optional[DashboardRewardsMap]
    updated_at: typing.Optional[datetime]

    @classmethod
    def from_proto(cls, message: dashboard_pb2.DashboardRewardsMap) -> 'DashboardItem':
        return cls(
            contract_id=message.contract_id,
            service=Services(message.service),
            active=message.active,
            rewards=DashboardRewardsMap.from_proto(message.rewards) if message.HasField('rewards') else None,
            updated_at=message.updated_at.ToDatetime() if message.HasField('updated_at') else None
        )

    def to_proto(self) -> dashboard_pb2.DashboardItem:
        return dashboard_pb2.DashboardItem(
            contract_id=self.contract_id,
            service=self.service.value,
            active=self.active,
            rewards=self.rewards.to_proto() if self.rewards else None,
            updated_at=timestamp_or_none(self.updated_at)
        )


@dataclass
class GetDashboardResponse(BaseStruct):
    dashboard: list[DashboardItem]

    @classmethod
    def from_proto(cls, message: dashboard_pb2.Dashboard) -> 'GetDashboardResponse':
        return cls(dashboard=[DashboardItem.from_proto(dashboard_item) for dashboard_item in message.dashboard])

    def to_proto(self) -> dashboard_pb2.Dashboard:
        return dashboard_pb2.Dashboard(dashboard=[dashboard_item.to_proto() for dashboard_item in self.dashboard])
