from dataclasses import dataclass, field
from datetime import datetime

from crm.agency_cabinet.common.proto_utils import BaseStruct, timestamp_or_none
from crm.agency_cabinet.rewards.proto import reports_pb2
from crm.agency_cabinet.common.consts import ReportsStatuses


@dataclass
class GetReportsInfoRequest(BaseStruct):
    agency_id: int
    filter_contract: int
    filter_service: str
    filter_type: str

    @classmethod
    def from_proto(cls, message: reports_pb2.GetReportsInfo) -> 'GetReportsInfoRequest':
        return cls(
            agency_id=message.agency_id,
            filter_contract=message.filter_contract,
            filter_service=message.filter_service,
            filter_type=message.filter_type
        )


@dataclass
class ReportInfo(BaseStruct):
    id: int
    contract_id: int
    name: str
    service: str
    type: str
    created_at: datetime
    period_from: datetime
    period_to: datetime
    status: str = ReportsStatuses.requested.value
    clients: list[int] = field(default_factory=list)

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportInfo) -> 'ReportInfo':
        return cls(
            id=message.id,
            contract_id=message.contract_id,
            name=message.name,
            service=message.service,
            type=message.type,
            created_at=message.created_at.ToDatetime() if message.HasField('created_at') else None,
            period_from=message.period_from.ToDatetime() if message.HasField('period_from') else None,
            period_to=message.period_to.ToDatetime() if message.HasField('period_to') else None,
            status=message.status,
            clients=message.clients_ids
        )

    def to_proto(self) -> reports_pb2.ReportInfo:
        created_at = timestamp_or_none(self.created_at)
        period_from = timestamp_or_none(self.period_from)
        period_to = timestamp_or_none(self.period_to)

        return reports_pb2.ReportInfo(
            id=self.id,
            contract_id=self.contract_id,
            name=self.name,
            service=self.service,
            type=self.type,
            created_at=created_at,
            period_from=period_from,
            period_to=period_to,
            status=self.status,
            clients_ids=self.clients

        )


class DetailedReportInfo(ReportInfo):  # сейчас никакой разницы нет
    def to_proto(self) -> reports_pb2.DetailedReportInfo:
        created_at = timestamp_or_none(self.created_at)
        period_from = timestamp_or_none(self.period_from)
        period_to = timestamp_or_none(self.period_to)

        return reports_pb2.DetailedReportInfo(
            id=self.id,
            contract_id=self.contract_id,
            name=self.name,
            service=self.service,
            type=self.type,
            created_at=created_at,
            period_from=period_from,
            period_to=period_to,
            status=self.status,
            clients_ids=self.clients
        )


@dataclass
class GetDetailedReportInfoResponse(BaseStruct):
    report: DetailedReportInfo

    @classmethod
    def from_proto(cls, message: reports_pb2.DetailedReportInfo) -> 'GetDetailedReportInfoResponse':
        return cls(report=DetailedReportInfo.from_proto(message))

    def to_proto(self) -> reports_pb2.DetailedReportInfo:
        return self.report.to_proto()


@dataclass
class GetDetailedReportInfoRequest(BaseStruct):
    agency_id: int
    report_id: int

    @classmethod
    def from_proto(cls, message: reports_pb2.GetDetailedReportInfo) -> 'GetDetailedReportInfoRequest':
        return cls(agency_id=message.agency_id,
                   report_id=message.report_id)


@dataclass
class GetReportsInfoResponse(BaseStruct):
    reports: list[ReportInfo]

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportsInfoList) -> 'GetReportsInfoResponse':
        return cls(reports=[ReportInfo.from_proto(report) for report in message.reports])

    def to_proto(self) -> reports_pb2.ReportsInfoList:
        return reports_pb2.ReportsInfoList(reports=[report.to_proto() for report in self.reports])


@dataclass
class DeleteReportRequest(BaseStruct):
    agency_id: int
    report_id: int

    @classmethod
    def from_proto(cls, message: reports_pb2.DeleteReport) -> 'DeleteReportRequest':
        return cls(agency_id=message.agency_id,
                   report_id=message.report_id)


@dataclass
class DeleteReportResponse(BaseStruct):
    is_deleted: bool

    @classmethod
    def from_proto(cls, message: reports_pb2.DeleteResponse) -> 'DeleteReportResponse':
        return cls(is_deleted=message.is_deleted)

    def to_proto(self) -> reports_pb2.DeleteResponse:
        return reports_pb2.DeleteResponse(is_deleted=self.is_deleted)


@dataclass
class CreateReportRequest(BaseStruct):
    agency_id: int
    contract_id: int
    name: str
    service: str
    type: str
    period_from: datetime
    period_to: datetime
    clients: list[int] = field(default_factory=list)

    @classmethod
    def from_proto(cls, message: reports_pb2.CreateReport) -> 'CreateReportRequest':
        return cls(agency_id=message.agency_id,
                   contract_id=message.contract_id,
                   name=message.name,
                   service=message.service,
                   type=message.type,
                   period_from=message.period_from.ToDatetime() if message.HasField('period_from') else None,
                   period_to=message.period_to.ToDatetime() if message.HasField('period_to') else None,
                   clients=message.clients_ids
                   )


@dataclass
class CreateReportResponse(BaseStruct):
    report: DetailedReportInfo

    @classmethod
    def from_proto(cls, message: reports_pb2.DetailedReportInfo) -> 'CreateReportResponse':
        return cls(report=DetailedReportInfo.from_proto(message))

    def to_proto(self) -> reports_pb2.DetailedReportInfo:
        return self.report.to_proto()


@dataclass
class GetReportUrlRequest(BaseStruct):
    agency_id: int
    report_id: int

    @classmethod
    def from_proto(cls, message: reports_pb2.GetReportUrl) -> 'GetReportUrlRequest':
        return cls(agency_id=message.agency_id, report_id=message.report_id)


@dataclass
class GetReportUrlResponse(BaseStruct):
    report_url: str

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportUrl) -> 'GetReportUrlResponse':
        return cls(report_url=message.url)

    def to_proto(self) -> reports_pb2.ReportUrl:
        return reports_pb2.ReportUrl(url=self.report_url)
