import typing

from dataclasses import dataclass
from datetime import datetime
from decimal import Decimal

from crm.agency_cabinet.common.proto_utils import (
    BaseStruct, timestamp_or_none, decimal_to_string, decimal_percent_to_string
)
from crm.agency_cabinet.rewards.proto import rewards_info_pb2


@dataclass
class GetRewardsInfoRequest(BaseStruct):
    agency_id: int
    filter_from: datetime
    filter_to: datetime
    filter_contract: int  # TODO: rename filter_* -> *
    filter_type: str  # TODO: rename, use enum
    filter_is_paid: typing.Optional[bool]

    @classmethod
    def from_proto(cls, message: rewards_info_pb2.GetRewardsInfo) -> 'GetRewardsInfoRequest':
        # where validate values for type???
        return cls(agency_id=message.agency_id,
                   filter_from=message.filter_from.ToDatetime() if message.HasField('filter_from') else None,
                   filter_to=message.filter_to.ToDatetime() if message.HasField('filter_to') else None,
                   filter_contract=message.filter_contract,
                   filter_type=message.filter_type,
                   filter_is_paid=message.filter_is_paid.value if message.HasField('filter_is_paid') else None)


@dataclass
class RewardInfo(BaseStruct):
    id: int
    contract_id: int
    type: str
    services: list[str]
    got_scan: bool
    got_original: bool
    is_accrued: bool
    is_paid: bool
    payment_date: typing.Optional[datetime]
    payment: Decimal
    period_from: datetime

    @classmethod
    def from_proto(cls, message: rewards_info_pb2.RewardInfo) -> 'RewardInfo':
        return cls(
            id=message.id,
            contract_id=message.contract_id,
            type=message.type,
            services=message.services,
            got_scan=message.got_scan,
            got_original=message.got_original,
            is_accrued=message.is_accrued,
            is_paid=message.is_paid,
            payment_date=message.payment_date.ToDatetime() if message.HasField('payment_date') else None,
            payment=Decimal(message.payment),
            period_from=message.period_from.ToDatetime()
        )

    def to_proto(self) -> rewards_info_pb2.RewardInfo:
        period_from = timestamp_or_none(self.period_from)
        payment_date = timestamp_or_none(self.payment_date)

        return rewards_info_pb2.RewardInfo(
            id=self.id,
            contract_id=self.contract_id,
            type=self.type,
            services=self.services,
            got_scan=self.got_scan,
            got_original=self.got_original,
            is_accrued=self.is_accrued,
            is_paid=self.is_paid,
            payment_date=payment_date,
            payment=decimal_to_string(self.payment),
            period_from=period_from
        )


@dataclass
class GetRewardsInfoResponse(BaseStruct):
    rewards: list[RewardInfo]

    @classmethod
    def from_proto(cls, message: rewards_info_pb2.RewardsInfoList) -> 'GetRewardsInfoResponse':
        return cls(rewards=[RewardInfo.from_proto(reward) for reward in message.rewards])

    def to_proto(self) -> rewards_info_pb2.RewardsInfoList:
        return rewards_info_pb2.RewardsInfoList(rewards=[reward.to_proto() for reward in self.rewards])


@dataclass
class GetDetailedRewardInfoRequest(BaseStruct):
    agency_id: int
    reward_id: int

    @classmethod
    def from_proto(cls, message: rewards_info_pb2.GetDetailedRewardInfo) -> 'GetDetailedRewardInfoRequest':
        return cls(agency_id=message.agency_id,
                   reward_id=message.reward_id)


@dataclass
class DetailedServiceInfo(BaseStruct):
    service: str
    revenue: typing.Optional[Decimal]
    currency: str
    reward_percent: typing.Optional[Decimal]
    accrual: Decimal
    error_message: typing.Optional[str]

    @classmethod
    def from_proto(cls, message: rewards_info_pb2.DetailedServiceInfo) -> 'DetailedServiceInfo':
        return cls(
            service=message.service,
            revenue=Decimal(message.revenue) if message.revenue else None,
            currency=message.currency,
            reward_percent=Decimal(message.reward_percent) if message.reward_percent else None,
            accrual=Decimal(message.accrual),
            error_message=message.error_message,
        )

    def to_proto(self) -> rewards_info_pb2.DetailedServiceInfo:
        return rewards_info_pb2.DetailedServiceInfo(
            service=self.service,
            revenue=decimal_to_string(self.revenue),
            currency=self.currency,
            reward_percent=decimal_percent_to_string(self.reward_percent),
            accrual=decimal_to_string(self.accrual),
            error_message=self.error_message,
        )


@dataclass
class DocumentInfo(BaseStruct):
    id: int
    name: str
    sending_date: datetime
    got_scan: bool
    got_original: bool

    @classmethod
    def from_proto(cls, message: rewards_info_pb2.DocumentInfo) -> 'DocumentInfo':
        return cls(
            id=message.id,
            name=message.name,
            sending_date=message.sending_date.ToDatetime() if message.HasField('sending_date') else None,
            got_scan=message.got_scan,
            got_original=message.got_original,
        )

    def to_proto(self) -> rewards_info_pb2.DocumentInfo:
        sending_date = timestamp_or_none(self.sending_date)

        return rewards_info_pb2.DocumentInfo(
            id=self.id,
            name=self.name,
            sending_date=sending_date,
            got_scan=self.got_scan,
            got_original=self.got_original
        )


@dataclass
class DetailedRewardInfo(BaseStruct):
    id: int
    contract_id: int
    type: str
    services: list[DetailedServiceInfo]
    documents: list[DocumentInfo]
    status: str
    accrual: typing.Optional[Decimal]
    payment: typing.Optional[Decimal]
    accrual_date: typing.Optional[datetime]
    payment_date: typing.Optional[datetime]
    period_from: datetime
    predict: bool

    @classmethod
    def from_proto(cls, message: rewards_info_pb2.DetailedRewardInfo) -> 'DetailedRewardInfo':
        return cls(
            id=message.id,
            contract_id=message.contract_id,
            type=message.type,
            services=[DetailedServiceInfo.from_proto(service) for service in message.services],
            documents=[DocumentInfo.from_proto(document) for document in message.documents],
            status=message.status,
            accrual=Decimal(message.accrual),
            payment=Decimal(message.payment),
            accrual_date=message.accrual_date.ToDatetime() if message.HasField('accrual_date') else None,
            payment_date=message.payment_date.ToDatetime() if message.HasField('payment_date') else None,
            period_from=message.period_from.ToDatetime() if message.HasField('period_from') else None,
            predict=message.predict,
        )

    def to_proto(self) -> rewards_info_pb2.DetailedRewardInfo:
        accrual_date = timestamp_or_none(self.accrual_date)
        payment_date = timestamp_or_none(self.payment_date)
        period_from = timestamp_or_none(self.period_from)

        return rewards_info_pb2.DetailedRewardInfo(
            id=self.id,
            contract_id=self.contract_id,
            type=self.type,
            services=[service.to_proto() for service in self.services],
            documents=[document.to_proto() for document in self.documents],
            status=self.status,
            accrual=decimal_to_string(self.accrual),
            payment=decimal_to_string(self.payment),
            accrual_date=accrual_date,
            payment_date=payment_date,
            period_from=period_from,
            predict=self.predict,
        )


@dataclass
class GetDetailedRewardInfoResponse(BaseStruct):
    reward: DetailedRewardInfo

    @classmethod
    def from_proto(cls, message: rewards_info_pb2.DetailedRewardInfo) -> 'GetDetailedRewardInfoResponse':
        return cls(reward=DetailedRewardInfo.from_proto(message))

    def to_proto(self) -> rewards_info_pb2.DetailedRewardInfo:
        return self.reward.to_proto()
