import logging
from datetime import datetime
from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder
from yt.wrapper import YtClient
from crm.agency_cabinet.common.consts import SERVICE_DISCOUNT_TYPE_MAP
from crm.agency_cabinet.rewards.server.src.celery.tasks.calculator.loader import BaseDataLoader, DataLoaderException

LOGGER = logging.getLogger('celery.tasks.calculator.actual.loader')


class ActualDataLoaderException(DataLoaderException):
    pass


class BaseActualDataLoader(BaseDataLoader):
    base_exception_class = ActualDataLoaderException
    COLUMNS = '''
        `dt` AS dt,
        `predict_amt` AS  revenue
    '''

    YQL_QUERY = '''
        use hahn;

        DECLARE $service_name AS String;
        DECLARE $predicts_table_path AS String;
        DECLARE $period_from AS String;
        DECLARE $period_to AS String;

        SELECT
            contract_id,
            AGGREGATE_LIST(
                AsStruct(
                    {}
                )
            ) AS indexes_and_grades


        FROM $predicts_table_path
        WHERE
            discount_name=$service_name
            AND dt<$period_to
            AND dt>=$period_from
        GROUP BY contract_id
        ;
    '''


class BaseActualDataWithEarlyPaymentLoader(BaseActualDataLoader):
    COLUMNS = '''
            `dt` AS dt,
            `rewards.predict_amt` AS  revenue,
            `early_payments.early_payment` AS early_payment
        '''

    YQL_QUERY = '''
        USE hahn;
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        DECLARE $actual_data_table_path AS String;
        DECLARE $early_payment_table_path AS String;
        DECLARE $period_from AS String;
        DECLARE $period_to AS String;
        DECLARE $discount_types AS List<Int64>;;
        DECLARE $service_name AS String;
        $date_parse = DateTime::Parse("%Y-%m-%dT%H:%M:%SZ");
        $early_payments = (SELECT
            contract_id,
            reward_to_charge * 50 AS early_payment,
            DateTime::MakeDate(DateTime::Update($date_parse(till_dt), NULl, NULL, 1)) AS parse_dt
        FROM $early_payment_table_path
        WHERE
            reward_type == 311
            AND discount_type in $discount_types
            AND till_dt >= $period_from
            AND till_dt < $period_to
        );
        $date_parse2 = DateTime::Parse("%Y-%m-%d");
        SELECT
            rewards.contract_id,
            AGGREGATE_LIST(
                AsStruct(
                    {}
                )
        ) AS indexes_and_grades
        FROM $actual_data_table_path AS rewards
        FULL JOIN $early_payments AS early_payments
            ON early_payments.contract_id == rewards.contract_id
            AND early_payments.parse_dt == DateTime::MakeDate($date_parse2(rewards.dt))
        WHERE
            rewards.dt >= $period_from
            AND rewards.dt < $period_to
            AND discount_name == $service_name
        GROUP BY rewards.contract_id
    ;
    '''

    def __init__(self, yt_client: YtClient, yql_client: YqlClient, actual_data_table_path: str,
                 early_payment_table_path: str, period_from: datetime, period_to: datetime):

        self.early_payment_table_path = early_payment_table_path
        super().__init__(yt_client, yql_client, actual_data_table_path, period_from, period_to)

    def build_yql_parameters(self):
        return {
            '$actual_data_table_path': YqlParameterValueBuilder.make_string(self.table_path),
            '$early_payment_table_path': YqlParameterValueBuilder.make_string(self.early_payment_table_path),
            '$period_from': YqlParameterValueBuilder.make_string(self.period_from.strftime('%Y-%m-%d')),
            '$period_to': YqlParameterValueBuilder.make_string(self.period_to.strftime('%Y-%m-%d')),
            '$service_name': YqlParameterValueBuilder.make_string(self.service_name),
            '$discount_types': YqlParameterValueBuilder.make_list(
                [
                    YqlParameterValueBuilder.make_int64(d) for d in SERVICE_DISCOUNT_TYPE_MAP[self.service]
                ]
            )
        }
