import logging
import typing

from datetime import datetime
from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder
from yt.wrapper import YtClient

from crm.agency_cabinet.common.yt.loaders import YqlLoader
from crm.agency_cabinet.rewards.server.src.celery.tasks.calculator.synchronizer import BaseCalculatorSynchronizer

LOGGER = logging.getLogger('celery.tasks.calculator.loader')


class DataLoaderException(Exception):
    pass


class BaseDataLoader(YqlLoader):
    service: str = None
    service_name: str = None
    synchronizer_class: typing.Type[BaseCalculatorSynchronizer] = None
    base_exception_class: typing.Type[DataLoaderException] = DataLoaderException
    COLUMNS = ''
    YQL_QUERY = ''

    def __init__(self, yt_client: YtClient, yql_client: YqlClient, table_path: str, period_from: datetime, period_to: datetime):
        if self.synchronizer_class is None:
            raise self.base_exception_class('Synchronizer class is undefined')

        self.service = self.synchronizer_class.service

        if self.service_name is None or self.service is None:
            LOGGER.error('Service is undefined')
            raise self.base_exception_class('Service is undefined')

        self.table_path = table_path
        self.period_from = period_from
        self.period_to = period_to

        super().__init__(
            yt_client, yql_client,
            yql_parameters=self.build_yql_parameters(),
            table_paths=[table_path],
        )

    def yql_query(self):
        return self.YQL_QUERY.format(self.COLUMNS)

    def build_yql_parameters(self):
        return {
            '$service_name': YqlParameterValueBuilder.make_string(self.service_name),
            '$predicts_table_path': YqlParameterValueBuilder.make_string(self.table_path),
            '$period_from': YqlParameterValueBuilder.make_string(self.period_from.strftime('%Y-%m-%d')),
            '$period_to': YqlParameterValueBuilder.make_string(self.period_to.strftime('%Y-%m-%d')),
        }

    async def _load(self, *args, **kwargs) -> bool:
        results = await self._get_results()
        data = results.table.get_iterator()
        return await self.synchronizer_class().process_data(data, prefer_actual=kwargs.get('prefer_actual', True))
