import logging

from datetime import datetime
from yt.wrapper import YtClient
from yql.api.v1.client import YqlClient

from crm.agency_cabinet.common.consts import END_FIN_YEAR_2022
from crm.agency_cabinet.common.consts.reward import BASE_PREDICTS_SOURCE_PATH
from crm.agency_cabinet.common.celery.base import async_to_sync
from crm.agency_cabinet.common.consts import CalculatorServiceType
from crm.agency_cabinet.rewards.server.config.clients import YT_CONFIG, YQL_CONFIG
from crm.agency_cabinet.rewards.server.src.celery.base import celery_app as celery
from crm.agency_cabinet.rewards.server.src.celery.tasks.calculator.predicts.aggregator.loader import \
    MetaAggregatorBasePredictsLoader, AggregatorBusinessPredictsLoader
from crm.agency_cabinet.rewards.server.src.celery.tasks.calculator.predicts.prof.loader import \
    ProfMediaPredictsLoader, ProfVideoPredictsLoader, ProfZenPredictsLoader, \
    ProfDirectPredictsLoader, ProfBusinessPredictsLoader
from crm.agency_cabinet.rewards.server.src.celery.tasks.calculator.predicts.base.loader import  \
    BaseZenPredictsLoader, BaseDirectPredictsLoader, BaseVideoPredictsLoader, \
    BaseMediaPredictsLoader, BaseBusinessPredictsLoader

LOGGER = logging.getLogger('celery.tasks.calculator.predicts.load_predicts_task')


@celery.task(bind=True)
def load_predicts_task(self, period_from: str = None, period_to: str = None, force_load: bool = False, prefer_actual: bool = True):
    if period_from:
        period_from = datetime.strptime(period_from, '%Y-%m-%d')
    else:
        now = datetime.now()
        period_from = datetime(year=now.year, month=now.month, day=1)

    if period_to:
        period_to = datetime.strptime(period_to, '%Y-%m-%d')
    else:
        period_to = END_FIN_YEAR_2022

    period_from = period_from.strftime('%Y-%m-%d')
    period_to = period_to.strftime('%Y-%m-%d')

    load_service_predicts_task.delay(CalculatorServiceType.direct.value, period_from, period_to, force_load, prefer_actual)
    load_service_predicts_task.delay(CalculatorServiceType.media.value, period_from, period_to, force_load, prefer_actual)
    load_service_predicts_task.delay(CalculatorServiceType.video.value, period_from, period_to, force_load, prefer_actual)
    load_service_predicts_task.delay(CalculatorServiceType.zen.value, period_from, period_to, force_load, prefer_actual)
    load_service_predicts_task.delay(CalculatorServiceType.business.value, period_from, period_to, force_load, prefer_actual)
    load_service_predicts_task.delay(CalculatorServiceType.aggregator.value, period_from, period_to, force_load, prefer_actual)


@celery.task(bind=True)
def load_service_predicts_task(self, service: str, period_from: str, period_to: str, force_load: bool, prefer_actual: bool):
    yt_client = YtClient(proxy='hahn', token=YT_CONFIG['TOKEN'])
    yql_client = YqlClient(token=YQL_CONFIG['TOKEN'])

    tables = yt_client.search(
        BASE_PREDICTS_SOURCE_PATH,
        node_type=['table'],
    )

    table_path = sorted(tables)[-1]

    period_from = datetime.strptime(period_from, '%Y-%m-%d')
    period_to = datetime.strptime(period_to, '%Y-%m-%d')

    @async_to_sync
    async def load():
        if service == CalculatorServiceType.media.value:
            await ProfMediaPredictsLoader(yt_client, yql_client, table_path, period_from, period_to).load(force_load=force_load, prefer_actual=prefer_actual)
            await BaseMediaPredictsLoader(yt_client, yql_client, table_path, period_from, period_to).load(force_load=force_load, prefer_actual=prefer_actual)

        elif service == CalculatorServiceType.video.value:
            await ProfVideoPredictsLoader(yt_client, yql_client, table_path, period_from, period_to).load(force_load=force_load, prefer_actual=prefer_actual)
            await BaseVideoPredictsLoader(yt_client, yql_client, table_path, period_from, period_to).load(force_load=force_load, prefer_actual=prefer_actual)

        elif service == CalculatorServiceType.zen.value:
            await ProfZenPredictsLoader(yt_client, yql_client, table_path, period_from, period_to).load(force_load=force_load, prefer_actual=prefer_actual)
            await BaseZenPredictsLoader(yt_client, yql_client, table_path, period_from, period_to).load(force_load=force_load, prefer_actual=prefer_actual)

        elif service == CalculatorServiceType.business.value:
            await ProfBusinessPredictsLoader(yt_client, yql_client, table_path, period_from, period_to).load(force_load=force_load, prefer_actual=prefer_actual)
            await BaseBusinessPredictsLoader(yt_client, yql_client, table_path, period_from, period_to).load(force_load=force_load, prefer_actual=prefer_actual)
            await AggregatorBusinessPredictsLoader(yt_client, yql_client, table_path, period_from, period_to).load(force_load=force_load, prefer_actual=prefer_actual)

        elif service == CalculatorServiceType.direct.value:
            await ProfDirectPredictsLoader(yt_client, yql_client, table_path, period_from, period_to).load(force_load=force_load, prefer_actual=prefer_actual)
            await BaseDirectPredictsLoader(yt_client, yql_client, table_path, period_from, period_to).load(force_load=force_load, prefer_actual=prefer_actual)

        elif service == CalculatorServiceType.aggregator.value:
            await MetaAggregatorBasePredictsLoader(yt_client, yql_client, table_path, period_from, period_to).load(force_load=force_load, prefer_actual=prefer_actual)

        else:
            LOGGER.error(f'Unknown service {service}')

    load()
