import logging
from yt.wrapper import YtClient

from crm.agency_cabinet.common.consts import Services, START_FIN_YEAR_2021, END_FIN_YEAR_2021, START_FIN_YEAR_2020, END_FIN_YEAR_2020
from crm.agency_cabinet.common.yt.base import MethodExtractor
from crm.agency_cabinet.rewards.common import structs
from crm.agency_cabinet.rewards.common.schemas import calculator as calculator_schemas
from crm.agency_cabinet.rewards.server.config.clients import YT_CONFIG
from crm.agency_cabinet.rewards.server.src.celery.base import celery_app as celery
from crm.agency_cabinet.rewards.server.src.db import models
from .base import EarlyPaymentDataLoader, PredictDataLoader, RevenueLoader, get_predict_start_from

LOGGER = logging.getLogger('celery.calculator.media')


class MediaRevenueLoader(RevenueLoader):
    service = Services.media.value

    def make_month_data(self, predict, period_from, revenue):
        return structs.CalculatorMonthData(
            indexes=[
                structs.CalculatorIndexData(
                    index_id=self.index_id,
                    revenue=revenue
                )
            ],
            period_from=period_from.replace(tzinfo=None),
            predict=predict
        )

    def make_data(self, months):
        data = structs.CalculatorData(months=months)
        return calculator_schemas.CalculatorDataSchema().dump(data)


class MediaEarlyPaymentDataLoader(EarlyPaymentDataLoader):
    service = Services.media.value

    def make_month_data(self, predict, period_from, index):
        return structs.CalculatorMonthData(
            period_from=period_from,
            predict=False,
            indexes=[index]
        )

    def make_data(self, months):
        data = structs.CalculatorData(months=months)
        return calculator_schemas.CalculatorDataSchema().dump(data)


class MediaPredictDataLoader(PredictDataLoader):
    service = Services.media.value

    INDEX_ID_TO_YT_COLUMN_MAP = {
        'revenue': 'predict_amt',
        'early_payment': 'predict_amt',
    }

    def _make_table_row(self, row):
        period_from = PredictDataLoader._get_period_from(row)
        contract_id = PredictDataLoader._get_contract_id(row)

        indexes = []
        for index_id, yt_column in MediaPredictDataLoader.INDEX_ID_TO_YT_COLUMN_MAP.items():
            if row[yt_column]:
                indexes.append(structs.CalculatorIndexData(
                    index_id=index_id,
                    revenue=row[yt_column]
                ))

        months = [structs.CalculatorMonthData(
            period_from=period_from,
            predict=True,
            indexes=indexes
        )]

        data = structs.CalculatorData(months=months)

        return {
            'contract_id': contract_id,
            'data': calculator_schemas.CalculatorDataSchema().dump(data)
        }


@celery.task(bind=True)
def load_calculator_media_prof_data_task(self):
    client_config = {
        'cluster': 'hahn',
        'token': YT_CONFIG['TOKEN'],
        'config': {}
    }

    MediaRevenueLoader(START_FIN_YEAR_2021, END_FIN_YEAR_2021, 'revenue').load()

    early_payment_data_loader = MediaEarlyPaymentDataLoader(
        period_from=START_FIN_YEAR_2021,
        table_path='//home/balance/prod/bo/v_ar_rewards',
        model=models.CalculatorData,
        columns_mapper={
            'contract_id': 'contract_id',
            'data': 'data'
        },
        default_columns={
            'service': Services.media.value
        },
        client_config=client_config
    )

    early_payment_data_loader.load()

    client = YtClient(proxy='hahn', token=YT_CONFIG['TOKEN'])
    predict_tables = client.search(
        '//home/search-research/ga/agency_rewards/forecast/v1/predict/amt/release',
        node_type=['table'],
    )

    predict_loader = MediaPredictDataLoader(
        period_from=get_predict_start_from(),
        period_to=END_FIN_YEAR_2021,
        table_path=sorted(predict_tables)[-1],
        model=models.CalculatorData,
        columns_mapper={
            'contract_id': MethodExtractor('_get_contract_id'),
            'data': 'data'
        },
        default_columns={
            'service': Services.media.value
        },
        client_config=client_config
    )

    predict_loader.load()

    MediaRevenueLoader(START_FIN_YEAR_2020, END_FIN_YEAR_2020, 'revenue_prev', prev=True).load()
