import datetime

from dateutil.relativedelta import relativedelta
from yt.wrapper import YtClient

from yql.client.parameter_value_builder import YqlParameterValueBuilder

from crm.agency_cabinet.common.consts import START_FIN_YEAR_2020, RewardsTypes
from crm.agency_cabinet.common.utils.obfuscate import obfuscate_data
from crm.agency_cabinet.common.yt.base import MethodExtractor, ObfuscateDictExtractor
from crm.agency_cabinet.rewards.server.config.clients import YT_CONFIG, YQL_CONFIG
from crm.agency_cabinet.rewards.server.config.data_loaders import LOADERS_SETTINGS
from crm.agency_cabinet.rewards.server.src.celery.base import celery_app as celery
from crm.agency_cabinet.rewards.server.src.db.models import ServiceReward, Reward
from .month import MonthRewardLoader, ProfMonthLoaderYTReduce, ProfMonthRewardLoader  # noqa
from .quarter import ProfQuarterRewardLoader, QuarterRewardLoader, BusinessQuarterRewardLoader, BusinessProfQuarterRewardLoader  # noqa
from .semiyear import ProfSemiyearRewardLoaderYTReduce, ProfSemiyearRewardLoader, SemiyearRewardLoader, SemiyearRewardLoaderYTReduce  # noqa
from .early_payment import EarlyPaymentRewardLoader # noqa
from .reward_payment import RewardPaymentRewardLoader


CLASS_MAP = {
    (RewardsTypes.month.value, True): ProfMonthRewardLoader,
    (RewardsTypes.month.value, False): MonthRewardLoader,
    (RewardsTypes.quarter.value, True): ProfQuarterRewardLoader,
    (RewardsTypes.quarter.value, False): QuarterRewardLoader,
    (RewardsTypes.semiyear.value, False): SemiyearRewardLoaderYTReduce,
    (RewardsTypes.semiyear.value, True): ProfSemiyearRewardLoaderYTReduce,
}


def loader_factory(
        table_path: str,
        reward_period: RewardsTypes,
        is_prof: bool,
        should_obfuscate: bool,
        raise_on_unknown_contract: bool,
        force_load: bool):
    # TODO: better
    cls = CLASS_MAP[(reward_period.value, is_prof)]
    base_column_mappers = {
        'reward_id': MethodExtractor('_get_reward_id'),
        'revenue': ObfuscateDictExtractor('amt', obfuscate_data) if should_obfuscate else 'amt',
        'discount_type': 'discount_type',
        'payment': ObfuscateDictExtractor('reward', obfuscate_data) if should_obfuscate else 'reward',
        'service': MethodExtractor('_get_service'),
        'reward_percent': MethodExtractor('_get_reward_percent')
    }

    return cls(
        table_path,
        ServiceReward,
        base_column_mappers,
        {},
        {
            'cluster': 'hahn',
            'token': YT_CONFIG['TOKEN'],
            'config': {}  # TODO: config?
        },
        raise_on_unknown_contract=raise_on_unknown_contract,
        force_load=force_load
    )


@celery.task(bind=True)
def load_single_month_table_task(self, table_path, is_prof=False, force_load=False):
    loader = loader_factory(table_path=table_path,
                            is_prof=is_prof,
                            reward_period=RewardsTypes.month,
                            should_obfuscate=LOADERS_SETTINGS['OBFUSCATE_SENSITIVE_DATA'],
                            raise_on_unknown_contract=not LOADERS_SETTINGS['CREATE_STUB_FOR_UNKNOWN_CONTRACTS'],
                            force_load=force_load)
    loader.load()


@celery.task(bind=True)
def load_single_quarter_table_task(self, table_path, is_prof=False, force_load=False):
    loader = loader_factory(table_path=table_path,
                            is_prof=is_prof,
                            reward_period=RewardsTypes.quarter,
                            should_obfuscate=LOADERS_SETTINGS['OBFUSCATE_SENSITIVE_DATA'],
                            raise_on_unknown_contract=not LOADERS_SETTINGS['CREATE_STUB_FOR_UNKNOWN_CONTRACTS'],
                            force_load=force_load)
    loader.load()


@celery.task(bind=True)
def load_single_semiyear_table_task(self, table_path, is_prof=False, force_load=False):
    loader = loader_factory(table_path=table_path,
                            is_prof=is_prof,
                            reward_period=RewardsTypes.semiyear,
                            should_obfuscate=LOADERS_SETTINGS['OBFUSCATE_SENSITIVE_DATA'],
                            raise_on_unknown_contract=not LOADERS_SETTINGS['CREATE_STUB_FOR_UNKNOWN_CONTRACTS'],
                            force_load=force_load)
    loader.load()


@celery.task(bind=True)
def load_month_rewards_tables_from_yt_task(self):
    client = YtClient(proxy='hahn',
                      token=YT_CONFIG['TOKEN'], )
    # TODO: сделать выбор базового пути как-нибудь лучше (конфиг, конфиг в базе????)
    for table in client.search(
        '//home/balance/prod/yb-ar/rewards/yandex/2021-prof_20-m-2021_m_prof',
        node_type=['table'],
    ):
        load_single_month_table_task.delay(table_path=table, is_prof=True)

    for table in client.search(
        '//home/balance/prod/yb-ar/rewards/yandex/2021-base-m-2021_m_base',
        node_type=['table'],
    ):
        # TODO: проверять нужна ли синхронизация здесь, чтобы не генерировать лишнюю таску?
        #       но будет не очень аккуратно
        load_single_month_table_task.delay(table_path=table, is_prof=False)


@celery.task(bind=True)
def load_quarter_rewards_tables_from_yt_task(self):
    client = YtClient(proxy='hahn',
                      token=YT_CONFIG['TOKEN'], )
    # TODO: сделать выбор базового пути как-нибудь лучше (конфиг, конфиг в базе????)
    for table in client.search(
        '//home/balance/prod/yb-ar/rewards/yandex/2021-prof_20-q-2021_q_prof',
        node_type=['table'],
    ):
        load_single_quarter_table_task.delay(table_path=table, is_prof=True)

    for table in client.search(
        '//home/balance/prod/yb-ar/rewards/yandex/2021-base-q-2021_q_base',
        node_type=['table'],
    ):
        load_single_quarter_table_task.delay(table_path=table, is_prof=False)


@celery.task(bind=True)
def load_semiyear_rewards_tables_from_yt_task(self):
    client = YtClient(proxy='hahn',
                      token=YT_CONFIG['TOKEN'], )
    # TODO: сделать выбор базового пути как-нибудь лучше (конфиг, конфиг в базе????)
    for table in client.search(
        '//home/balance/prod/yb-ar/rewards/yandex/2021-2021-prof_20-hy-2021_hy_prof-hf',
        node_type=['table'],
    ):
        load_single_semiyear_table_task.delay(table_path=table, is_prof=True)

    for table in client.search(
        '//home/balance/prod/yb-ar/rewards/yandex/2021-2021-base-hy-2021_hy_base-hf',
        node_type=['table'],
    ):
        load_single_semiyear_table_task.delay(table_path=table, is_prof=False)


@celery.task(bind=True)
def load_early_payment_rewards_table_from_yt_task(self):
    table_path = '//home/balance/prod/bo/v_ar_rewards'
    should_obfuscate = LOADERS_SETTINGS['OBFUSCATE_SENSITIVE_DATA']  # TODO: fix obfuscate in updates

    column_mappers = {
        'reward_id': MethodExtractor('_get_reward_id'),
        'discount_type': 'discount_type',
        'payment': ObfuscateDictExtractor('reward_to_charge', obfuscate_data) if should_obfuscate else 'reward_to_charge',
        # or reward_to_charge as in calculator???
        'service': MethodExtractor('_get_service'),
        'reward_percent': MethodExtractor('_get_reward_percent')
    }

    loader = EarlyPaymentRewardLoader(
        table_path,
        ServiceReward,
        column_mappers,
        {},
        {
            'cluster': 'hahn',
            'token': YT_CONFIG['TOKEN'],
            'config': {}  # TODO: config?
        },
        raise_on_unknown_contract=not LOADERS_SETTINGS['CREATE_STUB_FOR_UNKNOWN_CONTRACTS'],
        # TODO: add force_load to task args
    )
    loader.load()


@celery.task(bind=True)
def load_business_rewards_for_march_july_task(self):
    should_obfuscate = LOADERS_SETTINGS['OBFUSCATE_SENSITIVE_DATA']
    column_mappers = {
        'reward_id': MethodExtractor('_get_reward_id'),
        'revenue': ObfuscateDictExtractor('amt', obfuscate_data) if should_obfuscate else 'amt',
        'discount_type': 'discount_type',
        'payment': ObfuscateDictExtractor('reward', obfuscate_data) if should_obfuscate else 'reward',
        'service': MethodExtractor('_get_service'),
        'reward_percent': MethodExtractor('_get_reward_percent')
    }

    client = YtClient(proxy='hahn', token=YT_CONFIG['TOKEN'])

    path_loader_map = {
        '//home/balance/prod/yb-ar/rewards/geo/2020-prof_20-m-prof_20_geo_subscription': ProfMonthRewardLoader,
        '//home/balance/prod/yb-ar/rewards/geo/2020-base-m-base_geo_subscription': MonthRewardLoader,
        '//home/balance/prod/yb-ar/rewards/geo/2020-base_light-m-base_light_geo_subscription': MonthRewardLoader,

        '//home/balance/prod/yb-ar/rewards/geo/2020-prof_20-q-prof_20_geo_subscription': BusinessProfQuarterRewardLoader,
        '//home/balance/prod/yb-ar/rewards/geo/2020-base-q-base_geo_subscription': BusinessQuarterRewardLoader,
        '//home/balance/prod/yb-ar/rewards/geo/2020-base_light-q-base_light_geo_subscription': BusinessQuarterRewardLoader,
    }

    for path, loader_class in path_loader_map.items():
        for table in client.search(path, node_type=['table']):
            loader = loader_class(
                table_path=table,
                model=ServiceReward,
                columns_mapper=column_mappers,
                default_columns={},
                client_config={
                    'cluster': 'hahn',
                    'token': YT_CONFIG['TOKEN'],
                    'config': {}  # TODO: config?
                }
            )
            loader.load()


@celery.task(bind=True)
def load_rewards_payments_table_from_yt_task(self, from_dt: str = None):
    if from_dt:
        from_dt = datetime.datetime.strptime(from_dt, '%Y-%m-%d')
    else:
        from_dt = START_FIN_YEAR_2020

    parameters = {
        '$payment_table_name': YqlParameterValueBuilder.make_string('//home/balance/prod/bo/v_ar_rewards'),
        '$period_from': YqlParameterValueBuilder.make_string((from_dt - relativedelta(hours=3)).strftime('%Y-%m-%d'))
    }

    loader = RewardPaymentRewardLoader(
        from_dt=from_dt,
        table_path='//home/balance/prod/bo/v_ar_rewards',
        model=Reward,
        columns_mapper={
            'payment': 'payment',
            'contract_id': 'contract_id'
        },
        default_columns={},
        client_config={
            'cluster': 'hahn',
            'token': YT_CONFIG['TOKEN'],
            'config': {}
        },
        force_load=True,
        yql_token=YQL_CONFIG['TOKEN'],
        yql_query=RewardPaymentRewardLoader.YQL_QUERY,
        yql_parameters=parameters
    )

    loader.load()
