import logging
import typing

from datetime import datetime, timezone
from sqlalchemy import and_

from crm.agency_cabinet.common.consts.service import service_id_to_name
from crm.agency_cabinet.common.db.models import BaseModel
from crm.agency_cabinet.common.yql.base import YqlModelLoader
from crm.agency_cabinet.common.yt.base import async_to_sync

from crm.agency_cabinet.rewards.server.src.db import models

LOGGER = logging.getLogger('celery.load_rewards')


class CommonRewardLoader(YqlModelLoader):
    def _get_loader_tag(self) -> str:
        return f'{super()._get_loader_tag()}_{self.reward_type.value}'

    def _preprocess_period_from(self, row) -> typing.Tuple[int, int, int]:
        pass

    def _extract_period_from(self, row) -> datetime:
        year, month, day = self._preprocess_period_from(row)
        return datetime(year=year, month=month, day=day, tzinfo=timezone.utc)


class BaseRewardLoader(CommonRewardLoader):
    IS_LOADING_PREDICTIONS = False
    contract_ids = {}

    def _before_start(self):
        @async_to_sync
        async def _list_contracts():
            return await models.Contract.select('id').gino.all()

        self.contract_ids = {contract.id for contract in _list_contracts()}

    def _find_duplicate(self, yt_row) -> typing.Optional[BaseModel]:
        @async_to_sync
        async def _delete_reward(contract_id: int, reward_type: str, period_from: datetime):
            async with self.db_bind:
                return await models.Reward.delete.where(
                    and_(
                        models.Reward.contract_id == contract_id,
                        models.Reward.type == reward_type,
                        models.Reward.period_from == period_from,
                    )
                ).gino.status(read_only=False, reuse=False)

        @async_to_sync
        async def _get_reward(contract_id: int, reward_type: str, period_from: datetime):
            async with self.db_bind:
                return await models.Reward.query.where(
                    and_(
                        models.Reward.contract_id == contract_id,
                        models.Reward.type == reward_type,
                        models.Reward.period_from == period_from,
                    )
                ).gino.first()

        reward = _get_reward(
            self._get_column_value(yt_row, 'contract_id'),
            self.reward_type.value,
            self._extract_period_from(yt_row)
        )

        if reward is not None and reward.predict and not self.IS_LOADING_PREDICTIONS:
            # if we are loading historical rewards, but already have prediction
            # we are removing this prediction => we have no duplicate more => returning None
            _delete_reward(
                self._get_column_value(yt_row, 'contract_id'),
                self.reward_type.value,
                self._extract_period_from(yt_row)
            )
            return None

        return reward

    def _process_duplicate(self, yt_row, db_row: BaseModel):
        @async_to_sync
        async def _update_reward(reward: models.Reward, payment):
            async with self.db_bind:
                return await reward.update(
                    payment=payment,
                    predict=self.IS_LOADING_PREDICTIONS,
                    is_accrued=not self.IS_LOADING_PREDICTIONS,
                ).apply()

        # if we already have historical reward, then we should keep it
        if not db_row.predict and self.IS_LOADING_PREDICTIONS:
            return

        _update_reward(
            db_row,
            self._get_column_value(yt_row, 'payment')
        )

    def _check_if_should_skip_row(self, yt_row) -> bool:
        contract_id = self._get_column_value(yt_row, 'contract_id')

        return contract_id not in self.contract_ids


class BaseServiceRewardLoader(CommonRewardLoader):
    IS_LOADING_PREDICTIONS = False

    def _before_start(self):
        self.reward_to_reward_id = {}

    def _extract_discount_type(self, row) -> int:
        pass

    def _extract_service_name(self, row) -> str:
        return service_id_to_name(self._extract_discount_type(row))

    def _extract_revenue(self, row) -> typing.Optional:
        return self._get_column_value(row, 'revenue')

    def _extract_reward_percent(self, row):
        return None

    def _extract_reward_id(self, row) -> int:
        @async_to_sync
        async def _get_reward_id(_contract_id: int, _reward_type: str, _period_from: datetime):
            async with self.db_bind:
                reward = await models.Reward.query.where(
                    and_(
                        models.Reward.contract_id == _contract_id,
                        models.Reward.type == _reward_type,
                        models.Reward.period_from == _period_from,
                        models.Reward.predict.is_(self.IS_LOADING_PREDICTIONS)
                    )
                ).gino.first()

                return reward.id if reward else None

        contract_id = self._get_column_value(row, 'contract_id')
        reward_type = self.reward_type.value
        period_from = self._extract_period_from(row)

        reward_id = self.reward_to_reward_id.get(
            (contract_id, reward_type, period_from)
        )
        if reward_id is not None:
            return reward_id

        reward_id = _get_reward_id(contract_id, reward_type, period_from)
        self.reward_to_reward_id[
            (contract_id, reward_type, period_from)
        ] = reward_id

        return reward_id

    def _find_duplicate(self, yt_row) -> typing.Optional[BaseModel]:
        @async_to_sync
        async def _get_service_reward(_reward_id: int, service: str, discount_type: int):
            async with self.db_bind:
                return await models.ServiceReward.query.where(
                    and_(
                        models.ServiceReward.reward_id == _reward_id,
                        models.ServiceReward.service == service,
                        models.ServiceReward.discount_type == discount_type
                    )
                ).gino.first()

        return _get_service_reward(
            self._extract_reward_id(yt_row),
            self._extract_service_name(yt_row),
            self._extract_discount_type(yt_row)
        )

    def _process_duplicate(self, yt_row, db_row: BaseModel):
        @async_to_sync
        async def _update_service_reward(service_reward: models.ServiceReward, payment):
            async with self.db_bind:
                return await service_reward.update(
                    payment=payment,
                    revenue=self._extract_revenue(yt_row),
                    reward_percent=self._extract_reward_percent(yt_row)
                ).apply()

        _update_service_reward(
            db_row,
            self._get_column_value(yt_row, 'payment'),
        )

    def _check_if_should_skip_row(self, yt_row) -> bool:
        return self._extract_reward_id(yt_row) is None
