from decimal import Decimal

from crm.agency_cabinet.common.db.models import BaseModel, db

from crm.agency_cabinet.rewards.server.src.db.models.reward import Reward
from crm.agency_cabinet.common.consts.service import Services


class ServiceReward(BaseModel):
    __tablename__ = 'service_reward'

    id = db.Column(db.BigInteger, primary_key=True)
    reward_id = db.Column(db.BigInteger, nullable=False)
    service = db.Column(db.Text, nullable=False)
    payment = db.Column(db.Numeric(precision=18, scale=6), nullable=True)
    revenue = db.Column(db.Numeric(precision=18, scale=6), nullable=True)
    reward_percent = db.Column(db.Numeric(precision=18, scale=6), nullable=True)
    discount_type = db.Column(db.Integer, nullable=False)

    __table_args__ = (
        db.ForeignKeyConstraint(['reward_id'], ['reward.id'], ondelete='CASCADE'),
        db.UniqueConstraint('reward_id', 'discount_type', 'service'),  # запрещаем у одной премии
                                                                       # быть нескольким discount_type
                                                                       # они должны группироваться при выгрузках данных
        db.CheckConstraint(
            f"""service in ({", ".join((f"'{e.value}'" for e in Services))})""",
            name='validate_services'
        )  # автогенерация checkConstraint еще не поддержана в alembic

    )

    _reward = None

    @property
    def raw_percent(self) -> Decimal:
        return 100 * self.payment / self.revenue if self.payment and self.revenue else Decimal(0)

    @property
    def reward(self) -> Reward:
        return self._reward

    @reward.setter
    def reward(self, value: Reward):
        self._reward = value
