import typing
import logging
from contextlib import AsyncExitStack
from aioboto3 import Session
from aiobotocore.config import AioConfig
from smb.common.rmq.rpc.server import BaseRpcHandler
from library.python.monlib.metric_registry import MetricRegistry
from crm.agency_cabinet.rewards.common import structs
from crm.agency_cabinet.rewards.proto import (
    common_pb2, request_pb2, rewards_info_pb2, contracts_info_pb2,
    calculator_pb2, reports_pb2, dashboard_pb2, documents_pb2
)
from crm.agency_cabinet.rewards.server.src import procedures
from crm.agency_cabinet.common.yadoc import YaDocClient
from crm.agency_cabinet.common.server.common.tvm import get_tvm_client
from crm.agency_cabinet.common.server.common.tvm import Tvm2Config
from crm.agency_cabinet.common.consts.tvm import TVMIdTest


LOGGER = logging.getLogger('rewards.handler')


class Handler(BaseRpcHandler):
    _request_proto = request_pb2.RpcRequest

    def __init__(
        self,
        mds_access_key_id: str = None,
        mds_secret_access_key: str = None,
        mds_endpoint_url: str = None,
        yadoc_endpoint_url: str = None,
        yadoc_tvm_id: int = None,
        tvm_config: Tvm2Config = None,
        metric_registry: MetricRegistry = None
    ):
        self.mds_access_key_id = mds_access_key_id
        self.mds_secret_access_key = mds_secret_access_key
        self.mds_endpoint_url = mds_endpoint_url
        self.yadoc_endpoint_url = yadoc_endpoint_url
        self.tvm_config = tvm_config
        self.boto3_session = None
        self.context_stack = None
        self.s3_resource = None
        self.s3_client = None
        self.tvm_client = None
        self.yadoc_client: typing.Optional[YaDocClient] = None
        self.yadoc_tvm_id = yadoc_tvm_id if yadoc_tvm_id is not None else TVMIdTest.yadoc.value
        self.metric_registry = metric_registry

    async def _setup_tvm_client(self) -> bool:
        if self.tvm_config:
            self.tvm_client = get_tvm_client(self.tvm_config)
            return True
        return False

    async def _setup_yadoc(self) -> bool:
        if self.yadoc_endpoint_url:
            self.yadoc_client = YaDocClient(
                self.yadoc_endpoint_url,
                tvm_client=self.tvm_client,
                yadoc_tvm_id=self.yadoc_tvm_id,
                raise_for_status=True
            )
            return True
        return False

    async def _setup_s3(self) -> bool:
        if self.mds_access_key_id and self.mds_secret_access_key and self.mds_endpoint_url:
            self.boto3_session = Session(
                aws_access_key_id=self.mds_access_key_id,
                aws_secret_access_key=self.mds_secret_access_key
            )
            self.context_stack = AsyncExitStack()
            self.s3_resource = await self.context_stack.enter_async_context(
                self.boto3_session.resource(
                    's3',
                    endpoint_url=self.mds_endpoint_url,
                    config=AioConfig(s3={'addressing_style': 'virtual'})
                )
            )
            self.s3_client = await self.context_stack.enter_async_context(
                self.boto3_session.client(
                    's3',
                    endpoint_url=self.mds_endpoint_url,
                    config=AioConfig(s3={'addressing_style': 'virtual'})
                )
            )
            return True
        return False

    async def setup(self) -> bool:
        success = await self._setup_tvm_client() and await self._setup_yadoc() and await self._setup_s3()
        LOGGER.info('Handler successfully init' if success else 'Failed to fully init handler')
        return success

    async def teardown(self):
        if self.context_stack is not None:
            await self.context_stack.aclose()
        if self.yadoc_client is not None:
            await self.yadoc_client.close_session()

    async def ping(self, _: common_pb2.Empty) -> common_pb2.PingOutput:
        return common_pb2.PingOutput(ping='pong')

    async def get_rewards_info(self, message: rewards_info_pb2.GetRewardsInfo) -> rewards_info_pb2.GetRewardsInfoOutput:
        result = await procedures.GetRewardsInfo()(request=structs.GetRewardsInfoRequest.from_proto(message))
        return rewards_info_pb2.GetRewardsInfoOutput(result=result.to_proto())

    async def get_detailed_reward_info(self, message: rewards_info_pb2.GetDetailedRewardInfo) -> rewards_info_pb2.GetDetailedRewardInfoOutput:
        try:
            result = await procedures.GetDetailedRewardInfo()(
                request=structs.GetDetailedRewardInfoRequest.from_proto(message)
            )
            return rewards_info_pb2.GetDetailedRewardInfoOutput(result=result.to_proto())
        except procedures.rewards_info.NoSuchReward:
            return rewards_info_pb2.GetDetailedRewardInfoOutput(no_such_reward=common_pb2.Empty())
        except procedures.rewards_info.UnsuitableAgency:
            return rewards_info_pb2.GetDetailedRewardInfoOutput(unsuitable_agency=common_pb2.Empty())

    async def get_contracts_info(self, message: contracts_info_pb2.GetContractsInfo) -> contracts_info_pb2.GetContractsInfoOutput:
        result = await procedures.GetContractsInfo()(request=structs.GetContractsInfoRequest.from_proto(message))
        return contracts_info_pb2.GetContractsInfoOutput(result=result.to_proto())

    async def get_calculator_meta(self, message: calculator_pb2.GetCalculatorMeta) -> calculator_pb2.GetCalculatorMetaOutput:
        try:
            result = await procedures.GetCalculatorMeta()(
                request=structs.GetCalculatorMetaRequest.from_proto(message)
            )
        except procedures.NoSuchContract:
            return calculator_pb2.GetCalculatorMetaOutput(no_such_contract=common_pb2.Empty())
        except procedures.BunkerNotFound:
            return calculator_pb2.GetCalculatorMetaOutput(not_found=common_pb2.Empty())
        except procedures.BunkerError as e:
            LOGGER.error('Something went wrong while fetching meta from Bunker: %s', e)
            return calculator_pb2.GetCalculatorMetaOutput(bunker_error=str(e))

        return calculator_pb2.GetCalculatorMetaOutput(result=result.result)

    async def get_calculator_data(
        self,
        message: calculator_pb2.GetCalculatorData
    ) -> calculator_pb2.GetCalculatorDataOutput:
        try:
            result = await procedures.GetCalculatorData()(
                request=structs.GetCalculatorDataRequest.from_proto(message)
            )
        except procedures.NoSuchCalculatorData:
            return calculator_pb2.GetCalculatorDataOutput(not_found=common_pb2.Empty())

        return calculator_pb2.GetCalculatorDataOutput(result=result.result)

    async def get_reports_info(self, message: reports_pb2.GetReportsInfo) -> reports_pb2.GetReportsInfoOutput:
        result = await procedures.GetReportsInfo()(request=structs.GetReportsInfoRequest.from_proto(message))
        return reports_pb2.GetReportsInfoOutput(result=result.to_proto())

    async def get_detailed_report_info(self, message: reports_pb2.GetDetailedReportInfo) -> reports_pb2.GetDetailedReportInfoOutput:
        try:
            result = await procedures.GetDetailedReportInfo()(
                request=structs.GetDetailedReportInfoRequest.from_proto(message)
            )
            return reports_pb2.GetDetailedReportInfoOutput(result=result.to_proto())
        except procedures.NoSuchReport:
            return reports_pb2.GetDetailedReportInfoOutput(no_such_report=common_pb2.Empty())
        except procedures.UnsuitableAgency:
            return reports_pb2.GetDetailedReportInfoOutput(unsuitable_agency=common_pb2.Empty())

    async def delete_report(self, message: reports_pb2.DeleteReport) -> reports_pb2.DeleteReportOutput:
        try:
            result = await procedures.DeleteReport()(request=structs.DeleteReportRequest.from_proto(message))
            return reports_pb2.DeleteReportOutput(result=result.to_proto())
        except procedures.NoSuchReport:
            return reports_pb2.DeleteReportOutput(no_such_report=common_pb2.Empty())
        except procedures.UnsuitableAgency:
            return reports_pb2.DeleteReportOutput(unsuitable_agency=common_pb2.Empty())

    async def create_report(self, message: reports_pb2.CreateReport) -> reports_pb2.CreateReportOutput:
        try:
            result = await procedures.CreateReport()(request=structs.CreateReportRequest.from_proto(message))
            return reports_pb2.CreateReportOutput(result=result.to_proto())
        except procedures.UnsuitableAgency:
            return reports_pb2.CreateReportOutput(unsuitable_agency=common_pb2.Empty())
        except procedures.NoSuchContract:
            return reports_pb2.CreateReportOutput(no_such_contract=common_pb2.Empty())
        except procedures.ImpossibleReport as ex:
            return reports_pb2.CreateReportOutput(unsupported_parameters=str(ex))

    async def get_report_url(self, message: reports_pb2.GetReportUrl) -> reports_pb2.GetReportUrlOutput:
        try:
            result = await procedures.GetReportUrl()(
                request=structs.GetReportUrlRequest.from_proto(message),
                s3_client=self.s3_client
            )
            return reports_pb2.GetReportUrlOutput(result=result.to_proto())
        except procedures.NoSuchReport:
            return reports_pb2.GetReportUrlOutput(no_such_report=common_pb2.Empty())
        except procedures.UnsuitableAgency:
            return reports_pb2.GetReportUrlOutput(unsuitable_agency=common_pb2.Empty())
        except procedures.FileNotFound:
            return reports_pb2.GetReportUrlOutput(file_not_found=common_pb2.Empty())
        except procedures.FileNotReady:
            return reports_pb2.GetReportUrlOutput(report_not_ready=common_pb2.Empty())

    async def get_dashboard(self, message: dashboard_pb2.GetDashboard):
        try:
            result = await procedures.GetDashboard()(
                request=structs.GetDashboard.from_proto(message)
            )
            return dashboard_pb2.DashboardOutput(result=result.to_proto())
        except procedures.UnknownYear:
            return dashboard_pb2.DashboardOutput(unsupported_parameters=common_pb2.Empty())

    async def get_document_url(self, message: documents_pb2.GetDocumentUrl) -> documents_pb2.GetDocumentUrlOutput:
        try:
            result = await procedures.GetDocumentUrl()(
                request=structs.GetDocumentUrlRequest.from_proto(message),
                yadoc_client=self.yadoc_client
            )
            return documents_pb2.GetDocumentUrlOutput(result=result.to_proto())
        except procedures.NoSuchReport:
            return documents_pb2.GetDocumentUrlOutput(no_such_document=common_pb2.Empty())
        except procedures.UnsuitableAgency:
            return documents_pb2.GetDocumentUrlOutput(unsuitable_agency=common_pb2.Empty())
        except procedures.FileNotFound:
            return documents_pb2.GetDocumentUrlOutput(file_not_found=common_pb2.Empty())

    async def list_available_calculators(self, message: calculator_pb2.ListAvailableCalculatorsInput):

        result = await procedures.ListAvailableCalculators()(
            request=structs.ListAvailableCalculatorsInput.from_proto(message),
        )
        return calculator_pb2.ListAvailableCalculatorsOutput(result=result.to_proto())
