import json

from dataclasses import dataclass
from sqlalchemy import and_
from crm.agency_cabinet.common.consts import ContractType, CalculatorServiceType, get_start_of_current_fin_year_with_dt
from crm.agency_cabinet.common.bunker import BunkerClient, BunkerError, BunkerNotFoundError
from crm.agency_cabinet.rewards.common import structs
from crm.agency_cabinet.rewards.server.config import BUNKER_SETTINGS
from crm.agency_cabinet.rewards.server.src.db import models, db

from .exceptions import BunkerError as RewardsBunkerError, BunkerNotFound as RewardsBunkerNotFound, NoSuchCalculatorData, NoSuchContract


@dataclass
class GetCalculatorMeta:
    client = BunkerClient(
        project='agency-cabinet', host=BUNKER_SETTINGS['HOST'], default_version=BUNKER_SETTINGS['VERSION']
    )
    BASE_NODE = 'calculator-meta'

    async def __call__(self, request: structs.GetCalculatorMetaRequest) -> structs.GetCalculatorMetaResponse:
        contract: models.Contract = await models.Contract.query.where(
            and_(
                models.Contract.agency_id == request.agency_id,
                models.Contract.id == request.contract_id,
            )
        ).gino.first()
        if contract is None:
            raise NoSuchContract(f'Can\'t find contract with {request.contract_id} id for agency {request.agency_id}')

        try:
            if contract.type == ContractType.base.value:
                node=f'{self.BASE_NODE}/{request.version}/{contract.type}/{contract.payment_type}/{request.service}'

            else:
                node=f'{self.BASE_NODE}/{request.version}/{contract.type}/{request.service}'

            result = await self.client.cat(node=node)

            return structs.GetCalculatorMetaResponse(
                result=json.dumps(result['data'])
            )
        except BunkerNotFoundError as e:
            raise RewardsBunkerNotFound(e.message)
        except BunkerError as e:
            raise RewardsBunkerError(e.message)


@dataclass
class GetCalculatorData:
    async def __call__(self, request: structs.GetCalculatorDataRequest) -> structs.GetCalculatorDataResponse:
        calculator_data = await models.CalculatorData.query.where(
            and_(
                models.CalculatorData.contract_id == request.contract_id,
                models.CalculatorData.service == request.service.value,
                models.CalculatorData.version == request.version
            )
        ).gino.first()

        if not calculator_data:
            raise NoSuchCalculatorData()

        return structs.GetCalculatorDataResponse(
            result=json.dumps(calculator_data.data)
        )


@dataclass
class ListAvailableCalculators:
    async def __call__(self, request: structs.ListAvailableCalculatorsInput) -> structs.ListAvailableCalculatorsResponse:

        contract_types = {i.value for i in ContractType}
        services = {i.value for i in CalculatorServiceType}

        query = db.select(
            [
                models.CalculatorData.contract_id,
                models.CalculatorData.service,
                models.CalculatorData.version,
                models.Contract.type.label('contract_type')
            ]

        ).select_from(
            models.CalculatorData.outerjoin(models.Contract)
        ).where(
            and_(
                models.Contract.agency_id == request.agency_id,
                models.Contract.type.in_(contract_types),
                models.CalculatorData.service.in_(services),
                models.Contract.finish_date > get_start_of_current_fin_year_with_dt()
            )
        ).order_by(
            models.CalculatorData.contract_id,
            models.CalculatorData.version,
            models.CalculatorData.service
        )

        if request.contract_id is not None:
            query = query.where(models.Contract.id == request.contract_id)

        calculator_info = await query.gino.all()

        return structs.ListAvailableCalculatorsResponse(
            calculators_descriptions=[
                structs.CalculatorDescription(
                    contract_id=info.contract_id,
                    service=CalculatorServiceType(info.service),
                    version=info.version,
                    contract_type=ContractType(info.contract_type)
                )
                for info in calculator_info
            ]
        )
