import typing
from dataclasses import dataclass
from datetime import timezone

from crm.agency_cabinet.rewards.common import structs
from crm.agency_cabinet.rewards.server.src.db import Contract, ReportMetaInfo, S3MdsFile
from crm.agency_cabinet.common.mds import create_presigned_url_async
from crm.agency_cabinet.common.mds.definitions import S3AsyncClient
from crm.agency_cabinet.rewards.server.src.celery.tasks.reports.generate_report import check_params
from crm.agency_cabinet.common.consts.report import ReportsStatuses

from .exceptions import NoSuchReport, UnsuitableAgency, NoSuchContract, ImpossibleReport, FileNotFound, FileNotReady


@dataclass
class GetReportsInfo:
    async def __call__(self, request: structs.GetReportsInfoRequest) -> structs.GetReportsInfoResponse:
        query = ReportMetaInfo.load(
            contract=Contract.on(
                ReportMetaInfo.contract_id == Contract.id)).where(
            Contract.agency_id == request.agency_id
        )
        if request.filter_type:
            query = query.where(ReportMetaInfo.type == request.filter_type)
        if request.filter_contract:
            query = query.where(ReportMetaInfo.contract_id == request.filter_contract)
        if request.filter_service:
            query = query.where(ReportMetaInfo.service == request.filter_service)

        reports: typing.List[ReportMetaInfo] = await query.gino.all()

        return structs.GetReportsInfoResponse(
            reports=[structs.ReportInfo(
                id=report.id,
                contract_id=report.contract_id,
                name=report.name,
                service=report.service,
                type=report.type,
                created_at=report.created_at,
                period_from=report.period_from,
                period_to=report.period_to,
                status=report.status,
                clients=report.clients_ids,
            ) for report in reports]
        )


@dataclass
class GetDetailedReportInfo:
    async def __call__(self, request: structs.GetDetailedReportInfoRequest) -> structs.GetDetailedReportInfoResponse:
        query = ReportMetaInfo.load(contract=Contract.on(
            ReportMetaInfo.contract_id == Contract.id)).query.where(ReportMetaInfo.id == request.report_id)

        report = await query.gino.first()

        if report is None:
            raise NoSuchReport
        if report.contract.agency_id != request.agency_id:
            raise UnsuitableAgency

        return structs.GetDetailedReportInfoResponse(report=structs.DetailedReportInfo(
            id=report.id,
            contract_id=report.contract_id,
            name=report.name,
            service=report.service,
            type=report.type,
            created_at=report.created_at,
            period_from=report.period_from,
            period_to=report.period_to,
            status=report.status,
            clients=report.clients_ids,
        ))


@dataclass
class CreateReport:
    async def __call__(self, request: structs.CreateReportRequest) -> structs.CreateReportResponse:
        contract = await Contract.query.where(
            Contract.id == request.contract_id,
        ).gino.first()

        if contract is None:
            raise NoSuchContract()

        if contract.agency_id != request.agency_id:
            raise UnsuitableAgency()
        period_from = request.period_from.replace(tzinfo=timezone.utc)  # TODO: разобраться с таймзоной
        period_to = request.period_to.replace(tzinfo=timezone.utc)
        errors = check_params(period_from, period_to, request.type)
        if errors:
            msg = ';'.join(errors)
            raise ImpossibleReport(msg)
        # TODO: check clients

        report = await ReportMetaInfo.create(
            contract_id=request.contract_id,
            name=request.name,
            service=request.service,
            type=request.type,
            period_from=period_from,
            period_to=period_to,
            clients_ids=request.clients,
        )

        return structs.CreateReportResponse(report=structs.DetailedReportInfo(
            id=report.id,
            contract_id=report.contract_id,
            name=report.name,
            service=report.service,
            type=report.type,
            created_at=report.created_at,
            period_from=report.period_from,
            period_to=report.period_to,
            status=report.status,
            clients=report.clients_ids,
        ))


@dataclass
class DeleteReport:
    async def __call__(self, request: structs.DeleteReportRequest) -> structs.DeleteReportResponse:
        report = await ReportMetaInfo.load(contract=Contract.on(
            ReportMetaInfo.contract_id == Contract.id)).query.where(ReportMetaInfo.id == request.report_id).gino.first()

        if report is None:
            raise NoSuchReport
        if report.contract.agency_id != request.agency_id:
            raise UnsuitableAgency

        await report.delete()

        return structs.DeleteReportResponse(
            is_deleted=True
        )


@dataclass
class GetReportUrl:
    async def __call__(
        self,
        request:
        structs.GetReportUrlRequest,
        s3_client: S3AsyncClient
    ) -> structs.GetReportUrlResponse:
        report: ReportMetaInfo = await ReportMetaInfo.load(
            contract=Contract.on(ReportMetaInfo.contract_id == Contract.id),
            file=S3MdsFile.on(ReportMetaInfo.file_id == S3MdsFile.id)
        ).query.where(ReportMetaInfo.id == request.report_id).gino.first()

        if report is None:
            raise NoSuchReport
        if report.contract.agency_id != request.agency_id:
            raise UnsuitableAgency
        if report.status != ReportsStatuses.ready.value:
            raise FileNotReady
        if report.file is None:
            raise FileNotFound

        url = await create_presigned_url_async(s3_client, report.file.bucket, report.file.name)
        return structs.GetReportUrlResponse(
            report_url=url
        )
