package ru.yandex.crm.apphost.kotlin.common.apphost

import mu.KotlinLogging
import ru.yandex.crm.apphost.kotlin.common.middleware.ApphostPathHandlerPipelineBuilder
import ru.yandex.crm.library.kotlin.service.Service
import ru.yandex.passport.tvmauth.TvmClient
import ru.yandex.web.apphost.api.AppHostService
import ru.yandex.web.apphost.api.AppHostServiceBuilder
import ru.yandex.web.apphost.api.grpc.AppHostTypedServant
import ru.yandex.web.apphost.api.servant.AppHostGrpcProperties

class AppHostService(
    config: AppHostConfig,
    pathHandlers: List<AppHostPathHandler>,
    typedPathHandlers: List<AppHostTypedServant>,
    tvmClient: TvmClient,
) : Service {
    companion object {
        private val logger = KotlinLogging.logger { }
    }

    private val service: AppHostService
    private var isStarted: Boolean = false

    init {
        val builder = AppHostServiceBuilder.forPort(config.port)

        val registeredPaths = mutableSetOf<String>()
        for (handler in pathHandlers) {
            logger.info("register path: ${handler.path}")
            if (registeredPaths.contains(handler.path)) {
                throw IllegalArgumentException("Path ${handler.path} already registered")
            }

            builder.withPathHandler(handler.path, handler)
            registeredPaths.add(handler.path)
        }

        val typedHandlers = typedPathHandlers
            .flatMap { it.handlers }
            .map { ApphostPathHandlerPipelineBuilder.createDefaultFor(it).build() }
        for (typedHandler in typedHandlers) {
            logger.info("register path: {}", typedHandler.path)
            if (registeredPaths.contains(typedHandler.path)) {
                throw IllegalAccessException("Path ${typedHandler.path} already registered")
            }

            builder.withPathHandler(typedHandler.path, typedHandler)
            registeredPaths.add(typedHandler.path)
        }

        val grpcProperties = AppHostGrpcProperties()
        grpcProperties.addInterceptor(AuthenticationInterceptor(tvmClient))
        builder.useGrpc(grpcProperties)

        service = builder.build()
    }

    override suspend fun start() {
        if (isStarted) {
            throw IllegalStateException("Service is started already")
        }

        service.start()
        isStarted = true
    }

    override suspend fun stop() {
        if (!isStarted) {
            throw IllegalStateException("Service is not started")
        }

        service.stop()
        isStarted = false
    }
}
